// This file is part of the Attempto Java Packages.
// Copyright 2008, Attempto Group, University of Zurich (see http://attempto.ifi.uzh.ch).
//
// The Attempto Java Packages is free software: you can redistribute it and/or modify it under the
// terms of the GNU Lesser General Public License as published by the Free Software Foundation,
// either version 3 of the License, or (at your option) any later version.
//
// The Attempto Java Packages is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public License along with the Attempto
// Java Packages. If not, see http://www.gnu.org/licenses/.

package ch.uzh.ifi.attempto.acewiki;

import java.io.IOException;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import ch.uzh.ifi.attempto.acewiki.core.ontology.Ontology;
import ch.uzh.ifi.attempto.ape.APELocal;

import nextapp.echo2.app.ApplicationInstance;
import nextapp.echo2.webcontainer.WebContainerServlet;

/**
 * This servlet class should be launched by the web server in order to start an AceWiki application. If you are using a
 * web application archive (WAR) then the web.xml file should contain something like this:
 * 
 * <pre>
 * &lt;servlet&gt;
 *   &lt;servlet-name&gt;GeoWiki&lt;/servlet-name&gt;
 *   &lt;servlet-class&gt;ch.uzh.ifi.attempto.acewiki.AceWikiServlet&lt;/servlet-class&gt;
 *   &lt;init-param&gt;
 *     &lt;param-name&gt;ontology&lt;/param-name&gt;
 *     &lt;param-value&gt;geo&lt;/param-value&gt;
 *   &lt;/init-param&gt;
 *   &lt;init-param&gt;
 *     &lt;param-name&gt;baseuri&lt;/param-name&gt;
 *     &lt;param-value&gt;http://attempto.ifi.uzh.ch/acewiki/&lt;/param-value&gt;
 *   &lt;/init-param&gt;
 *   &lt;init-param&gt;
 *     &lt;param-name&gt;title&lt;/param-name&gt;
 *     &lt;param-value&gt;Geography Wiki&lt;/param-value&gt;
 *   &lt;/init-param&gt;
 *   &lt;init-param&gt;
 *     &lt;param-name&gt;description&lt;/param-name&gt;
 *     &lt;param-value&gt;This wiki contains geographical knowledge.&lt;/param-value&gt;
 *   &lt;/init-param&gt;
 *   &lt;init-param&gt;
 *     &lt;param-name&gt;login&lt;/param-name&gt;
 *     &lt;param-value&gt;no&lt;/param-value&gt;
 *   &lt;/init-param&gt;
 * &lt;/servlet&gt;
 * </pre>
 * 
 * The parameter 'ontology' is mandatory and specifies the name of the ontology. This name is used in the URIs of the
 * OWL statements and for the file names on the server. The 'baseuri' parameter defines the base URI for the
 * OWL statements. The values of the parameters 'title' and 'description' are shown on the start page of the wiki. Finally,
 * the 'login' parameter defines whether users have to login before they can use the wiki. Note that the login feature
 * is implemented only very rudimentary.
 *<p>
 * Furthermore, SWI Prolog needs to be installed on the server and you need to have a compiled version of the Attempto APE
 * distribution. See the documentation of {@link APELocal} for more information. The server has to know the command of your
 * SWI Prolog installation and the location of the ape.exe file. This is done by context parameters in the web.xml file:
 * 
 * <pre>
 * &lt;context-param&gt;
 *   &lt;param-name&gt;prologcommand&lt;/param-name&gt;
 *   &lt;param-value&gt;swipl&lt;/param-value&gt;
 * &lt;/context-param&gt;
 *  
 * &lt;context-param&gt;
 *   &lt;param-name&gt;apecommand&lt;/param-name&gt;
 *   &lt;param-value&gt;/path/to/file/ape.exe&lt;/param-value&gt;
 * &lt;/context-param&gt;
 * </pre>
 * 
 * One last tiny thing: The server delay window (that is shown on the client when the server is busy) looks for the
 * wait icon "../wait.gif". You should copy the file "src/ch/uzh/ifi/attempto/acewiki/core/gui/img/wait.gif" to the
 * respective folder on the server. (This should be improved, I admit.)
 * 
 * @author Tobias Kuhn
 */
public class AceWikiServlet extends WebContainerServlet {
	
	private static final long serialVersionUID = -7342857942059126499L;
	
	private Ontology ontology;
	
	/**
	 * Creates a new AceWiki servlet object.
	 */
	public AceWikiServlet() {
	}
	
	public ApplicationInstance newApplicationInstance() {
		String ontologyName = getInitParameter("ontology");
		String baseURI = getInitParameter("baseuri");
		String title = getInitParameter("title");
		String description = getInitParameter("description");
		boolean login = "yes".equals(getInitParameter("login"));
		
		if (!APELocal.isInitialized()) {
			String prologCommand = getServletContext().getInitParameter("prologcommand");
			String apeCommand = getServletContext().getInitParameter("apecommand");
			if (prologCommand == null) prologCommand = "swipl";
			if (apeCommand == null) apeCommand = "ape.exe";
			APELocal.init(prologCommand, apeCommand);
		}
		
		if (ontology == null) {
			ontology = Ontology.loadOntology(ontologyName, baseURI);
		}
		
		Logger.log("syst", "syst", 0, "appl", "new application instance: " + ontologyName);
		return new AceWikiApp(ontology, title, description, login);
	}
	
	protected void process(HttpServletRequest request, HttpServletResponse response) throws IOException, ServletException {
		try {
			super.process(request, response);
		} catch (RuntimeException ex) {
			Logger.log("syst", "syst", 0, "fail", "fatal error: " + ex);
			ex.printStackTrace();
			throw ex;
		} catch (IOException ex) {
			Logger.log("syst", "syst", 0, "fail", "fatal error: " + ex);
			ex.printStackTrace();
			throw ex;
		} catch (ServletException ex) {
			Logger.log("syst", "syst", 0, "fail", "fatal error: " + ex);
			ex.printStackTrace();
			throw ex;
		}
	}
	
	// This was a try to support specific URLs for each wiki articles:
	/*
	protected void process(HttpServletRequest request, HttpServletResponse response) throws IOException, ServletException {
		String pageParameter = request.getParameter("page");
		if (app != null && pageParameter != null) {
			final OntologyElement oe = app.getWiki().getOntology().get(request.getParameter("page"));
			if (taskQueue == null) {
				taskQueue = app.createTaskQueue();
			}
			if (oe != null) {
				app.enqueueTask(taskQueue, new Runnable() {
					public void run() {
						app.getWiki().showPage(oe);
					}
				});
				try {wait(500);} catch (Exception ex) {}
			}
		}
		
		super.process(request, response);
	}
	*/

}