// This file is part of the Attempto Java Packages.
// Copyright 2008, Attempto Group, University of Zurich (see http://attempto.ifi.uzh.ch).
//
// The Attempto Java Packages is free software: you can redistribute it and/or modify it under the
// terms of the GNU Lesser General Public License as published by the Free Software Foundation,
// either version 3 of the License, or (at your option) any later version.
//
// The Attempto Java Packages is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public License along with the Attempto
// Java Packages. If not, see http://www.gnu.org/licenses/.

package ch.uzh.ifi.attempto.acewiki.core.ontology;

import java.util.ArrayList;
import java.util.List;

import ch.uzh.ifi.attempto.ape.Gender;
import ch.uzh.ifi.attempto.ape.LexiconEntry;

/**
 * This class represents an individual (in logic called "constant"). AceWiki supports only
 * proper names to represent individuals (and no extensions are planned in this respect).
 * For that reason, this class is proper name specific.
 *<p>
 * Proper names can be used with a definite article (e.g. "the Nile") or without (e.g. "Egypt").
 *<p>
 * Proper names have two word forms. The first one is the proper name with the definite
 * article if there is one. The second one is just the bare proper name. In the case of
 * proper names that do not use a definite article, both word forms are identical.
 *<p>
 * 0: proper name, preceded by "the" if used with definite article.
 * 1: bare proper name.
 *<p>
 * Examples: ["the Nile", "Nile"]; ["Egypt", "Egypt"].
 * 
 * @author Tobias Kuhn
 */
public class Individual extends OntologyElement {
	
	private String word;
	private boolean hasDefiniteArticle;
	
	private List<Concept> conceptsCache;
	private long conceptsCacheStateID = -1;
	
	/**
	 * Creates a new individual that has no name yet and is not registered to an ontology.
	 */
	public Individual() {
		this.word = "";
		this.hasDefiniteArticle = false;
	}
	
	public String[] getWords() {
		if (hasDefiniteArticle) {
			return new String[] {"the " + word, word};
		} else {
			return new String[] {word, word};
		}
	}
	
	public String getHeadword() {
		return getPrettyWord(1);
	}
	
	protected void changeWords(String... words) {
		if (words.length == 1) {
			word = words[0];
			hasDefiniteArticle = false;
		} else {
			word = words[1];
			hasDefiniteArticle = words[0].startsWith("the ");
		}
	}

	List<LexiconEntry> getLexiconEntries() {
		List<LexiconEntry> entries = new ArrayList<LexiconEntry>();
		if (hasDefiniteArticle) {
			entries.add(LexiconEntry.createPropernameDefSgEntry(word, word, Gender.NEUTRAL));
		} else {
			entries.add(LexiconEntry.createPropernameSgEntry(word, word, Gender.NEUTRAL));
		}
		return entries;
	}
	
	public String getType() {
		return "Proper Name";
	}
	
	public String getInternalType() {
		return "propername";
	}
	
	/**
	 * Returns true if the proper name has to be used with the definite article "the".
	 * 
	 * @return true if the definite article "the" has to be used.
	 */
	public boolean hasDefiniteArticle() {
		return hasDefiniteArticle;
	}
	
	public String getURISuffix() {
		return "#" + word;
	}
	
	/**
	 * Returns all concepts this individual belongs to.
	 * 
	 * @return A list of all concepts of this individual.
	 * @see Ontology#getConcepts(Individual)
	 */
	public List<Concept> getConcepts() {
		Ontology o = getOntology();
		if (conceptsCacheStateID != o.getStateID()) {
			conceptsCache = o.getConcepts(this);
			conceptsCacheStateID = o.getStateID();
		}
		return new ArrayList<Concept>(conceptsCache);
	}

	/**
	 * Returns true if the concepts of this individual are cached and do not have
	 * to be recalculated.
	 * 
	 * @return true if the concepts are cached.
	 */
	public boolean areConceptsCached() {
		return conceptsCacheStateID == getOntology().getStateID();
	}

}
