// This file is part of the Attempto Java Packages.
// Copyright 2008, Attempto Group, University of Zurich (see http://attempto.ifi.uzh.ch).
//
// The Attempto Java Packages is free software: you can redistribute it and/or modify it under the
// terms of the GNU Lesser General Public License as published by the Free Software Foundation,
// either version 3 of the License, or (at your option) any later version.
//
// The Attempto Java Packages is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public License along with the Attempto
// Java Packages. If not, see http://www.gnu.org/licenses/.

package ch.uzh.ifi.attempto.acewiki.gui.editor;

import java.util.ArrayList;

import nextapp.echo2.app.event.ActionEvent;
import nextapp.echo2.app.event.ActionListener;
import ch.uzh.ifi.attempto.acewiki.Task;
import ch.uzh.ifi.attempto.acewiki.Wiki;
import ch.uzh.ifi.attempto.acewiki.core.ontology.OntologyElement;
import ch.uzh.ifi.attempto.acewiki.core.ontology.Sentence;
import ch.uzh.ifi.attempto.acewiki.gui.page.ArticlePage;
import ch.uzh.ifi.attempto.echocomp.MessageWindow;
import ch.uzh.ifi.attempto.preditor.PreditorWindow;
import ch.uzh.ifi.attempto.preditor.text.TextContainer;
import ch.uzh.ifi.attempto.preditor.text.TextElement;

/**
 * This class manages the predictive editor. It creates the editor window and handles its
 * responses.
 * 
 * @author Tobias Kuhn
 */
public class SentenceEditorHandler implements ActionListener {
	
	private static final long serialVersionUID = -2083910385095284075L;
	
	private PreditorWindow editorWindow;
	private ArticlePage page;
	private Wiki wiki;
	private boolean edit;
	private Sentence sentence;
	
	private SentenceEditorHandler(Sentence sentence, ArticlePage page, boolean edit) {
		this.page = page;
		this.wiki = page.getWiki();
		this.edit = edit;
		this.sentence = sentence;
		AceWikiMenuCreator menuCreator = new AceWikiMenuCreator(wiki, page.getOntologyElement(), this);
		editorWindow = new PreditorWindow("Sentence Editor", wiki.getGrammar(), menuCreator);
		editorWindow.addActionListener(this);
		
		if (edit) {
			String t = sentence.getPrettyText();
			// remove the last element (i.e. the period '.' or question mark '?'):
			editorWindow.addText(t.substring(0, t.length()-1));
		}
	}
	
	/**
	 * Generates a new preditive editor (short "preditor") window to create a new sentence.
	 * 
	 * @param followingSentence The sentence in front of which the new sentences should be added,
	 *   or null if the sentences should be added to the end of the article.
	 * @param page The host page into which the sentence should be added.
	 * @return A new preditor window.
	 */
	public static PreditorWindow generatePreditorAddWindow(Sentence followingSentence, ArticlePage page) {
		SentenceEditorHandler h = new SentenceEditorHandler(followingSentence, page, false);
		return h.getPreditorWindow();
	}
	
	/**
	 * Generates a new preditive editor (short "preditor") window to create a new sentence.
	 * 
	 * @param sentence The sentence that should be edited.
	 * @param page The host page which contains the sentence to be edited.
	 * @return A new preditor window.
	 */
	public static PreditorWindow generatePreditorEditWindow(Sentence sentence, ArticlePage page) {
		SentenceEditorHandler h = new SentenceEditorHandler(sentence, page, true);
		return h.getPreditorWindow();
	}
	
	private PreditorWindow getPreditorWindow() {
		return editorWindow;
	}

	public void actionPerformed(ActionEvent e) {
		if (e.getSource() == editorWindow && e.getActionCommand().equals("OK")) {
			final TextContainer textContainer = editorWindow.getTextContainer();
			
			TextElement finalElement = editorWindow.getPossibleNextToken(".");
			if (finalElement == null) {
				finalElement = editorWindow.getPossibleNextToken("?");
			}
			if (finalElement != null) textContainer.addElement(finalElement);
			
			ArrayList<TextElement> l = textContainer.getTextElements();
			if (l.isEmpty() || l.get(l.size() - 1).getText().matches("[.?]")) {
				Task task = new Task() {
					
					int success;
					
					public void run() {
						OntologyElement el = page.getOntologyElement();
						if (edit) {
							wiki.log("edit", "sentence updated: " + textContainer.getText());
							success = el.edit(sentence, Sentence.generateSentences(textContainer, el));
						} else {
							wiki.log("edit", "sentence created: " + textContainer.getText());
							success = el.add(sentence, Sentence.generateSentences(textContainer, el));
						}
					}
					
					public void updateGUI() {
						page.update();
						if (success == 1) {
							wiki.showWindow(
								new MessageWindow(
									"Conflict",
									"A sentence is in conflict with the current knowledge. For that reason, " +
										"it cannot be added to the knowledge base.",
									"OK"
								)
							);
						} else if (success == 2) {
							wiki.showWindow(
								new MessageWindow(
									"Error",
									"A sentence could not be added to the knowledge base because the knowledge " +
										"base got too complex.",
									"OK"
								)
							);
						}
						if (page != null) {
							page.update();
							page.getWiki().update();
						}
					}
					
				};
				
				if (edit) {
					wiki.enqueueTaskShowingWaitWindow("Updating", "The knowledge base is being updated...", task);
				} else {
					wiki.enqueueTaskShowingWaitWindow("Updating", "The sentence is being added to the knowledge base...", task);
				}
				
				editorWindow.setVisible(false);
				
			} else {
				wiki.log("edit", "error: unfinished sentences");
				MessageWindow mw = new MessageWindow("Error", "There are unfinished sentences.", editorWindow, "OK");
				page.getWiki().showWindow(mw);
			}
		} else if (e.getSource() == editorWindow && e.getActionCommand().equals("Cancel")) {
			editorWindow.setVisible(false);
			editorWindow.dispose();
		} else if (e.getSource() instanceof TextElement) {
			editorWindow.addTextElement((TextElement) e.getSource());
		}
	}

}
