// This file is part of the Attempto Java Packages.
// Copyright 2008, Attempto Group, University of Zurich (see http://attempto.ifi.uzh.ch).
//
// The Attempto Java Packages is free software: you can redistribute it and/or modify it under the
// terms of the GNU Lesser General Public License as published by the Free Software Foundation,
// either version 3 of the License, or (at your option) any later version.
//
// The Attempto Java Packages is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public License along with the Attempto
// Java Packages. If not, see http://www.gnu.org/licenses/.

package ch.uzh.ifi.attempto.ape;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;

import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.httpclient.NameValuePair;
import org.apache.commons.httpclient.methods.PostMethod;

// TODO Why don't we use GET instead of POST (as least for shorter queries)?

/**
 * This class provides an interface to the Attempto Parsing Engine (APE) webservice
 * (i.e. HTTP server).
 * The HTTP server implementation is provided by <code>ape.exe</code>. To start
 * a server, execute for example:
 * 
 * <pre>
 * ./ape.exe -httpserver -port 8000
 * </pre>
 * 
 * @author Kaarel Kaljurand
 * @author Tobias Kuhn
 */
public class APEWebservice extends ACEParser {

	private String wsUrl;
	private NameValuePair uri = new NameValuePair("uri", "");
	private NameValuePair text = new NameValuePair("text", "");
	private NameValuePair ulextext = new NameValuePair("ulextext", "");
	private NameValuePair solo = new NameValuePair("solo", "owlxml");
	private NameValuePair guess = new NameValuePair("guess", "off");
	private NameValuePair noclex = new NameValuePair("noclex", "off");

	/**
	 * Creates a new parser object based on the URL of a running APE webservice.
	 * 
	 * @param wsUrl The URL of the APE webservice.
	 */
	public APEWebservice(String wsUrl) {
		this.wsUrl = wsUrl;
	}

	/**
	 * Sets the URL of the APE webservice to be used.
	 * 
	 * @param wsUrl The URL of the APE webservice.
	 */
	public void setWebserviceUrl(String wsUrl) {
		this.wsUrl = wsUrl;
	}

	public String getSoloOutput(String aceText, Lexicon lexicon, OutputType outputType) throws ACEParserException {
		setText(aceText);
		setUlex(lexicon);
		setSolo(outputType);
		return checkForErrors(getParserResponseAsString(new NameValuePair[] {text, ulextext, solo, uri, guess, noclex}));
	}


	public ACEParserResult getMultiOutput(String aceText, Lexicon lexicon, OutputType... outputTypes) {
		NameValuePair[] params = new NameValuePair[outputTypes.length + 5];
		setText(aceText);
		setUlex(lexicon);
		params[0] = text;
		params[1] = ulextext;
		params[2] = uri;
		params[3] = guess;
		params[4] = noclex;
		int counter = 5; 
		for (OutputType t : outputTypes) {
			params[counter] = new NameValuePair(t.toMultiFlag(), "on");
			counter++;
		}
		return new ACEParserResult(getParserResponseAsString(params));
	}


	public void setURI(String uri) {
		super.setURI(uri);
		this.uri.setValue(getURI());
	}


	public void setGuessingEnabled(boolean guessing) {
		super.setGuessingEnabled(guessing);
		if (isGuessingEnabled()) {
			this.guess.setValue("on");
		}
		else {
			this.guess.setValue("off");
		}
	}


	public void setClexEnabled(boolean clexEnabled) {
		super.setClexEnabled(clexEnabled);

		if (isClexEnabled()) {
			this.noclex.setValue("off");
		}
		else {
			this.noclex.setValue("on");
		}
	}


	private void setUlex(Lexicon lexicon) {
		if (lexicon == null) {
			this.ulextext.setValue("");
		}
		else {
			this.ulextext.setValue(lexicon.toString());
		}
	}

	private void setText(String text) {
		this.text.setValue(text);
	}


	private void setSolo(OutputType solo) {
		this.solo.setValue(solo.toSoloFlag());
	}

	private String getParserResponseAsString(NameValuePair[] parameters) {
		HttpClient client = new HttpClient();
		PostMethod method = new PostMethod(wsUrl);
		method.setRequestBody(parameters);
		String response = null;

		try {
			int statusCode = client.executeMethod(method);

			if (statusCode == HttpStatus.SC_OK) {
				response = inputStreamAsString(method.getResponseBodyAsStream());
			}
			else {
				throw new RuntimeException("HTTP request failed: " + method.getStatusLine() + ": " + wsUrl);
			}
		} catch (Exception e) {
			throw new RuntimeException("Accessing APE webservice failed: " + e.getMessage());
		} finally {
			method.releaseConnection();
		}

		return response;
	}

	private String inputStreamAsString(InputStream stream) throws IOException {
		BufferedReader br = new BufferedReader(new InputStreamReader(stream));
		StringBuffer sb = new StringBuffer();
		String line = null;

		while ((line = br.readLine()) != null) {
			sb.append(line);
			sb.append('\n');
		}

		br.close();
		return sb.toString();
	}
}
