% This file is part of the Attempto Java Packages.
% Copyright 2008, Attempto Group, University of Zurich (see http://attempto.ifi.uzh.ch).
%
% The Attempto Java Packages is free software: you can redistribute it and/or modify it under the
% terms of the GNU Lesser General Public License as published by the Free Software Foundation,
% either version 3 of the License, or (at your option) any later version.
%
% The Attempto Java Packages is distributed in the hope that it will be useful, but WITHOUT ANY
% WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
% PURPOSE. See the GNU Lesser General Public License for more details.
%
% You should have received a copy of the GNU Lesser General Public License along with the Attempto
% Java Packages. If not, see http://www.gnu.org/licenses/.


:- module(generate_java, [
		generate_java/4  % +InputFile, +Package, +Class, +StartCat
	]).

/** <module> Generate Java Grammar

This module transfroms from the Attempto Chartparser Grammar Notation (ACGN) into Prolog DCG grammars.
Usage:

==
swipl -s generate_java.pl -g "generate_java('acgn.pl', 'my.package', 'MyGrammar', 'my-start-cat')" -t halt
==

You may have to replace "swipl" by the name of your SWI Prolog executable.

@author Tobias Kuhn
@version 2008-05-25
*/

:- op(1200, xfx, '=>').
:- op(1200, xfx, '~>').
:- op(600, xfx, ':').
:- op(500, fx, '#').

:- style_check(-atom).


%% generate_java(+InputFile, +Package, +Class, +StartCat)
%
% This predicate reads the ACGN grammar from the input file and transforms it into a Java class that contains the grammar.

generate_java(InputFile, Package, Class, StartCat) :-
    open(InputFile, read, In),
    atom_concat(Class, '.java', OutputFile),
    open(OutputFile, write, Out),
    part(1, Part1),
    part(2, Part2),
    replace(Part1, '*package*', Package, Part1T1),
    replace(Part1T1, '*class*', Class, Part1T2),
    replace(Part1T2, '*startcat*', StartCat, Part1T3),
    write(Out, Part1T3),
    (
    	process(In, Out)
    ;
    	write(user_error, 'Error during parsing.\n')
    ),
    close(In),
    write(Out, Part2),
    close(Out).


replace(Input, Search, Replace, Ouput) :-
	concat_atom(Split, Search, Input),
	concat_atom(Split, Replace, Ouput).


process(In, Out) :-
	read_term(In, Term, [module(generate_java)]),
	( Term == end_of_file ->
		true
	;
		numbervars(Term, 0, _),
		process_term(Out, Term),
		process(In, Out)
	).


process_term(Out, Head => Body) :-
    !,
    write(Out, '\t\tl.clear();\n'),
    write(Out, '\t\tfeatureHash.clear();\n'),
    process_cat(Out, Head),
    process_cats(Out, Body),
    write(Out, '\t\taddRule(new Rule(l, true));\n').

process_term(Out, Head ~> Body) :-
    !,
    write(Out, '\t\tl.clear();\n'),
    write(Out, '\t\tfeatureHash.clear();\n'),
    process_cat(Out, Head),
    process_cats(Out, Body),
    write(Out, '\t\taddRule(new Rule(l, false));\n').

process_term(_, Term) :-
    format(user_error, 'WARNING. Cannot process term: ~q\n', Term).


process_cats(Out, Body) :-
    Body = ','(Cat, BodyRest),
    !,
    process_cat(Out, Cat),
    process_cats(Out, BodyRest).

process_cats(_, []) :-
    !.

process_cats(Out, BodyLast) :-
    process_cat(Out, BodyLast).


process_cat(_, []) :-
    !.

process_cat(Out, [T|Rest]) :-
    !,
    format(Out, '\t\tc = new Terminal("~w");\n', T),
    write(Out, '\t\tl.add(c);\n'),
    process_cat(Out, Rest).

process_cat(Out, # Cat) :-
    !,
    Cat =.. [Name|Features],
    format(Out, '\t\tc = new Terminal("~w");\n', Name),
    process_features(Out, Features),
    write(Out, '\t\tl.add(c);\n').

process_cat(Out, Cat) :-
    Cat =.. [Name|Features],
    format(Out, '\t\tc = new Nonterminal("~w");\n', Name),
    process_features(Out, Features),
    write(Out, '\t\tl.add(c);\n').


process_features(_, []).

process_features(Out, [Name:'$VAR'(N)|Rest]) :-
    !,
    format(Out, '\t\tsetFeature(c, "~w", ~w, featureHash);\n', [Name, N]),
    process_features(Out, Rest).

process_features(Out, [Name:Value|Rest]) :-
    !,
    format(Out, '\t\tc.setFeature("~w", new StringRef("~w"));\n', [Name, Value]),
    process_features(Out, Rest).


part(1, 'package *package*;

import java.util.ArrayList;
import java.util.HashMap;

import ch.uzh.ifi.attempto.chartparser.Nonterminal;
import ch.uzh.ifi.attempto.chartparser.Rule;
import ch.uzh.ifi.attempto.chartparser.StringRef;
import ch.uzh.ifi.attempto.chartparser.Terminal;
import ch.uzh.ifi.attempto.chartparser.Category;

/**
 * This grammar class is automatically generated by the SWI Prolog program "generate_java.pl".
 */
@SuppressWarnings("all")
public class *class* extends ch.uzh.ifi.attempto.chartparser.Grammar {
	
	/**
	 * Creates a new grammar object.
	 */
	public *class*() {
		super("*startcat*");
		ArrayList<Category> l = new ArrayList<Category>();
		Category c;
		HashMap<Integer, StringRef> featureHash = new HashMap<Integer, StringRef>();
		
').


part(2, '
	}
	
	private void setFeature(Category c, String featureName, int varID, HashMap<Integer, StringRef> featureHash) {
		if (featureHash.get(varID) == null) {
			StringRef stringRef = new StringRef();
			c.setFeature(featureName, stringRef);
			featureHash.put(varID, stringRef);
		} else {
			c.setFeature(featureName, featureHash.get(varID));
		}
	}
	
}').

