// This file is part of the Attempto Java Packages.
// Copyright 2008, Attempto Group, University of Zurich (see http://attempto.ifi.uzh.ch).
//
// The Attempto Java Packages is free software: you can redistribute it and/or modify it under the
// terms of the GNU Lesser General Public License as published by the Free Software Foundation,
// either version 3 of the License, or (at your option) any later version.
//
// The Attempto Java Packages is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public License along with the Attempto
// Java Packages. If not, see http://www.gnu.org/licenses/.

package ch.uzh.ifi.attempto.preditor.text;

import ch.uzh.ifi.attempto.chartparser.Terminal;

/**
 * This is a general purpose implementation of a text element.
 * 
 * @author Tobias Kuhn
 */
public class BasicTextElement implements TextElement {
	
	private final String originalText;
	private String text, category;
	private boolean capitalize;
	
	/**
	 * Creates a new basic text element.
	 * 
	 * @param text The text.
	 * @param category The category.
	 * @param capitalize true if the word should be capitalized in sentence-initial position.
	 */
	public BasicTextElement(String text, String category, boolean capitalize) {
		this.text = text;
		this.originalText = text;
		this.category = category;
		this.capitalize = capitalize;
	}
	
	/**
	 * Creates a new basic text element. It is capitalized in sentence-initial position.
	 * 
	 * @param text The text.
	 * @param category The category.
	 */
	public BasicTextElement(String text, String category) {
		this(text, category, true);
	}
	
	/**
	 * Creates a new basic text element containing a number. The text element is capitalized in
	 * sentence-initial position.
	 * 
	 * @param number The number.
	 * @param category The category.
	 */
	public BasicTextElement(int number, String category) {
		this(number + "", category, true);
	}
	
	/**
	 * Creates a new basic text element where the text is the name of the category.
	 * 
	 * @param category The category which is the same as the text.
	 * @param capitalize true if the word should be capitalized in sentence-initial position.
	 */
	public BasicTextElement(String category, boolean capitalize) {
		this(category, category, capitalize);
	}
	
	/**
	 * Creates a new basic text element where the text is the name of the category.
	 * The text element is capitalized in sentence-initial position.
	 * 
	 * @param category The category which is the same as the text.
	 */
	public BasicTextElement(String category) {
		this(category, category, true);
	}
	
	/**
	 * Creates a new basic text element containing a number which is also the name of the category.
	 * The text element is capitalized in sentence-initial position.
	 * 
	 * @param number The number.
	 */
	public BasicTextElement(int number) {
		this(number + "", number + "", true);
	}
	
	/**
	 * Creates a new basic text element on the basis of a terminal category object. The text is the
	 * same as the category name.
	 * 
	 * @param terminal The terminal category object.
	 * @param capitalize true if the word should be capitalized in sentence-initial position.
	 */
	public BasicTextElement(Terminal terminal, boolean capitalize) {
		this(terminal.getName(), terminal.getName(), capitalize);
	}
	
	/**
	 * Creates a new basic text element on the basis of a terminal category object. The text is the
	 * same as the category name. The text element is capitalized in sentence-initial position.
	 * 
	 * @param terminal The terminal category object.
	 */
	public BasicTextElement(Terminal terminal) {
		this(terminal.getName(), terminal.getName(), true);
	}
	
	public String getCategory() {
		return category;
	}
	
	public String getText() {
		return text;
	}
	
	public void checkNeighborTextElements(TextElement precedingTextElement, TextElement nextTextElement) {
		if ((precedingTextElement == null || precedingTextElement.getText().matches("(\\.|\\?|\\!)")) && capitalize) {
			String f = originalText.substring(0, 1);
			text = f.toUpperCase() + originalText.substring(1);
		} else {
			text = originalText;
		}
		
		if (nextTextElement != null && text.matches("(A|a)n?")) {
			String nt = nextTextElement.getText().toLowerCase();
			boolean an = false;
			if (nt.matches("[aeiou].*")) an = true;
			if (nt.matches("[fhlmnrsx]")) an = true;
			if (nt.matches("[fhlmnrsx]-.*")) an = true;
			if (nt.equals("u")) an = false;
			if (nt.matches("u-.*")) an = false;
			if (nt.matches("u[rtn]i.*")) an = false;
			if (nt.matches("use.*")) an = false;
			if (nt.matches("uk.*")) an = false;
			
			if (an) {
				text = text.substring(0, 1) + "n";
			} else {
				text = text.substring(0, 1);
			}
		}
	}
	
	public boolean equals(Object obj) {
		if (obj instanceof BasicTextElement) {
			BasicTextElement other = (BasicTextElement) obj;
			return (this.text.equals(other.text) && this.category.equals(other.category));
		}
		return false;
	}

}
