// This file is part of the Attempto Java Packages.
// Copyright 2008, Attempto Group, University of Zurich (see http://attempto.ifi.uzh.ch).
//
// The Attempto Java Packages is free software: you can redistribute it and/or modify it under the
// terms of the GNU Lesser General Public License as published by the Free Software Foundation,
// either version 3 of the License, or (at your option) any later version.
//
// The Attempto Java Packages is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public License along with the Attempto
// Java Packages. If not, see http://www.gnu.org/licenses/.

package ch.uzh.ifi.attempto.acewiki.core.ontology;

import static ch.uzh.ifi.attempto.ape.OutputType.DRSPP;
import static ch.uzh.ifi.attempto.ape.OutputType.OWLRDF;
import static ch.uzh.ifi.attempto.ape.OutputType.OWLXML;
import static ch.uzh.ifi.attempto.ape.OutputType.PARAPHRASE1;
import static ch.uzh.ifi.attempto.ape.OutputType.SYNTAX;
import static ch.uzh.ifi.attempto.ape.OutputType.SYNTAXPP;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.semanticweb.owl.model.OWLOntology;
import org.semanticweb.owl.model.OWLOntologyCreationException;

import ch.uzh.ifi.attempto.acewiki.core.text.OntologyTextElement;
import ch.uzh.ifi.attempto.acewiki.core.text.TextElemFactory;
import ch.uzh.ifi.attempto.ape.ACEParserResult;
import ch.uzh.ifi.attempto.ape.APELocal;
import ch.uzh.ifi.attempto.ape.Lexicon;
import ch.uzh.ifi.attempto.ape.LexiconEntry;
import ch.uzh.ifi.attempto.ape.MessageContainer;
import ch.uzh.ifi.attempto.preditor.text.BasicTextElement;
import ch.uzh.ifi.attempto.preditor.text.TextContainer;
import ch.uzh.ifi.attempto.preditor.text.TextElement;

/**
 * This class represents an ACE sentence which is either a declarative statement or a question.
 * Some declarative sentences can be translated into OWL and can participate in reasoning. Other
 * sentences have no OWL representation and do not participate in reasoning.
 *<p>
 * Each sentence belongs to exactly one article of an ontology element (the owner).
 *<p>
 * Parsing of the sentence is done lasily, i.e. at the first time when a parsing result is required.
 * Parsing fails silently. No exceptions are thrown if a sentence is not ACE compliant.
 * 
 * @author Tobias Kuhn
 */
public class Sentence {
	
	private String text;
	private Ontology ontology;
	private OntologyElement owner;
	private boolean integrated = false;
	
	// These fields are evaluated lazily:
	private TextContainer textContainer;
	private ACEParserResult parserResult;
	private String owlxml;
	private Boolean reasonerParticipant;
	private Boolean isOWL;
	private Boolean isOWLSWRL;
	private OWLOntology owlOntology;
	
	private List<Individual> answerCache;
	private long answerCacheStateID = -1;
	
	/**
	 * Creates a new asserted sentence. Asserted sentences must have an owner.
	 * 
	 * @param text The sentence text.
	 * @param owner The owner ontology element.
	 */
	public Sentence(String text, OntologyElement owner) {
		this.text = text;
		this.ontology = null;
		this.owner = owner;
	}
	
	/**
	 * Creates a new inferred sentence. Inferred sentence have no owner.
	 * 
	 * @param text The sentence text.
	 * @param ontology The ontology.
	 */
	public Sentence(String text, Ontology ontology) {
		this.text = text;
		this.ontology = ontology;
		this.owner = null;
	}
	
	/**
	 * Generates sentence objects out of a text container.
	 * 
	 * @param textContainer The text container.
	 * @param owner The owner ontology element of the sentences.
	 * @return A list of sentences.
	 */
	public static List<Sentence> generateSentences(TextContainer textContainer, OntologyElement owner) {
		ArrayList<Sentence> l = new ArrayList<Sentence>();
		TextContainer c = new TextContainer();
		for (TextElement e : textContainer.getTextElements()) {
			c.addElement(e);
			if (e.getText().matches("[.?]")) {
				l.add(new Sentence(getUnderscoredText(c), owner));
				c = new TextContainer();
			}
		}
		return l;
	}
	
	/**
	 * Loads a sentence from a serialized form.
	 * 
	 * @param serializedSentence The serialized sentence as a string.
	 * @param owner The owner ontology element of the sentence.
	 * @return A sentence object.
	 */
	static Sentence loadSentence(String serializedSentence, OntologyElement owner) {
		Sentence sentence = new Sentence(serializedSentence.substring(2), owner);
		sentence.setIntegrated(serializedSentence.charAt(0) == '|');
		return sentence;
	}
	
	private Ontology getOntology() {
		if (ontology == null) {
			ontology = owner.getOntology();
		}
		return ontology;
	}
	
	/**
	 * Returns a list of text elements that represent the tokens of this sentence.
	 * 
	 * @return A token list.
	 */
	public List<TextElement> getTextElements() {
		if (textContainer == null) {
			tokenize();
		}
		return textContainer.getTextElements();
	}
	
	/**
	 * Returns the owner ontology element of this sentence.
	 * 
	 * @return The owner ontology element.
	 */
	public OntologyElement getOwner() {
		return owner;
	}
	
	/**
	 * Returns the sentence text as a string. Underscores are used for compound words,
	 * e.g. "credit_card".
	 * 
	 * @return The sentence text as a string.
	 */
	public String getText() {
		if (textContainer == null) {
			tokenize();
		}
		return getUnderscoredText(textContainer);
	}
	
	/**
	 * Returns the sentence text as a string with underscores displayed as blanks. Compound
	 * words containing underscores like "credit_cards" are pretty-printed with blank characters:
	 * "credit card".
	 * 
	 * @return The sentence text as a pretty-printed string.
	 */
	public String getPrettyText() {
		return textContainer.getText();
	}
	
	/**
	 * Returns the parser result object.
	 * 
	 * @return The parser result object.
	 */
	public ACEParserResult getParserResult() {
		if (parserResult == null) {
			parse();
		}
		return parserResult;
	}
	
	/**
	 * Returns the OWL/XML representation of this sentence as a string.
	 * 
	 * @return The OWL/XML representation.
	 */
	public String getOWLXML() {
		if (owlxml == null) {
			parse();
		}
		return owlxml;
	}
	
	/**
	 * Returns true if this sentence participates in reasoning.
	 * 
	 * @return true if this sentence participates in reasoning.
	 */
	public boolean isReasonerParticipant() {
		if (reasonerParticipant == null) {
			parse();
		}
		return reasonerParticipant;
	}
	
	/**
	 * Returns true if this sentence has an OWL representation.
	 * 
	 * @return true if this sentence has an OWL representation.
	 */
	public boolean isOWL() {
		if (isOWL == null) {
			parse();
		}
		return isOWL;
	}
	
	/**
	 * Returns true if this sentence has an OWL or SWRL representation.
	 * 
	 * @return true if this sentence has an OWL or SWRL representation.
	 */
	public boolean isOWLSWRL() {
		if (isOWLSWRL == null) {
			parse();
		}
		return isOWLSWRL;
	}
	
	/**
	 * Returns the OWL ontology object that contains the OWL representation of this
	 * sentence.
	 * 
	 * @return The OWL ontology object.
	 * @throws OWLOntologyCreationException If the OWL ontology object creation failed.
	 */
	public OWLOntology getOWLOntology() throws OWLOntologyCreationException {
		if (owlOntology == null) {
			owlOntology = getOntology().readOWLOntology(getOWLXML());
		}
		return owlOntology;
	}
	
	/**
	 * Tokenizes the sentence text. A text container object is created.
	 */
	private void tokenize() {
		textContainer = new TextContainer();
		
		String t = text;
		t = t.replaceAll(" ", "&");
		t = t.replaceAll("\\.", "&.&");
		t = t.replaceAll("\\?", "&?&");
		t = t.replaceAll("&of", " of");
		t = t.replaceAll("&by", " by");
		
		ArrayList<String> tokens = new ArrayList<String>(Arrays.asList(t.split("&")));
		
		while (tokens.contains("")) {
			tokens.remove("");
		}
		
		toString();
		
		for (String s : tokens) {
			if (s.startsWith("<")) {
				try {
					long oeId = new Long(s.substring(1, s.indexOf(",")));
					int wordNumber = new Integer(s.substring(s.indexOf(",")+1, s.indexOf(">")));
					OntologyElement oe = getOntology().get(oeId);
					textContainer.addElement(TextElemFactory.createTextElement(oe, wordNumber));
				} catch (Exception ex) {
					throw new RuntimeException("Could not resolve link: " + s);
				}
			} else {
				OntologyElement oe = getOntology().get(s);
				if (oe == null) {
					textContainer.addElement(new BasicTextElement(s));
				} else if (oe instanceof Individual && ((Individual) oe).hasDefiniteArticle()) {
					textContainer.removeLastElement();
					textContainer.addElement(TextElemFactory.createTextElement(oe, 0));
				} else {
					// not 100% clean solution (several word forms of the same word can be identical):
					int wordId = Arrays.asList(oe.getWords()).indexOf(s);
					textContainer.addElement(TextElemFactory.createTextElement(oe, wordId));
				}
			}
		}
	}
	
	/**
	 * Parses the sentence text. The OWL and SWRL representations are calculated if possible.
	 * This method is called automatically the first time a parsing result is needed.
	 * Furthermore, it needs to be called each time a word form of an ontology element
	 * (that occurs in the sentence) has changed.
	 */
	synchronized void parse() {
		APELocal.getInstance().setURI(getOntology().getURI());
		APELocal.getInstance().setClexEnabled(false);
		Lexicon lexicon = new Lexicon();
		for (TextElement te : getTextElements()) {
			if (te instanceof OntologyTextElement) {
				OntologyElement oe = ((OntologyTextElement) te).getOntologyElement();
				for (LexiconEntry le : oe.getLexiconEntries()) {
					lexicon.addEntry(le);
				}
			}
		}
		parserResult = APELocal.getInstance().getMultiOutput(getText(), lexicon, PARAPHRASE1, SYNTAX, SYNTAXPP, OWLXML, OWLRDF, DRSPP);
		MessageContainer mc = parserResult.getMessageContainer();
		owlxml = parserResult.get(OWLXML);
		if (owlxml != null) {
			// Every OWL ontology object needs its own URI:
			long hashCode = (long) getText().hashCode() - Integer.MIN_VALUE;
			owlxml = owlxml.replace("URI=\"" + ontology.getURI() + "\">", "URI=\"" + ontology.getURI() + "/" + hashCode + "\">");
		}
		reasonerParticipant =
			(mc.getMessages("owl").size() == 0) &&
			(owlxml.indexOf("<swrl:Imp>") < 0) &&
			(owlxml.indexOf("<ObjectExistsSelf>") < 0) &&
			(owlxml.indexOf("<TransitiveObjectProperty>") < 0) &&
			(owlxml.length() > 0);
		if (isQuestion()) {
			reasonerParticipant = false;
		}
		isOWL =
			(mc.getMessages("owl").size() == 0) &&
			(owlxml.indexOf("<swrl:Imp>") < 0) &&
			(owlxml.length() > 0);
		isOWLSWRL =
			(mc.getMessages("owl").size() == 0) &&
			(owlxml.length() > 0);
		String messages = mc.toString();
		if (messages.length() > 0) {
			System.err.println("Parser messages: " + messages);
		}
		owlOntology = null;
	}
	
	/**
	 * This method tries to reassert a sentence that is not yet integrated. This is
	 * used for sentences that have an OWL representation but the integration failed
	 * because it introduced an inconsistency. Later, when the ontology has changed,
	 * the integration might succeed.
	 * 
	 * @return An integer value denoting the success/failure of the operation.
	 * @see Ontology#commitSentence(Sentence)
	 */
	public int reassert() {
		return getOntology().commitSentence(this);
	}
	
	/**
	 * Returns true if the sentence is integrated into the ontology.
	 * 
	 * @return true if the sentence is integrated into the ontology.
	 */
	public boolean isIntegrated() {
		return integrated;
	}
	
	void setIntegrated(boolean integrated) {
		this.integrated = integrated;
	}
	
	/**
	 * Returns true if the sentence is a question.
	 * 
	 * @return true if the sentence is a question.
	 */
	public boolean isQuestion() {
		return text.substring(text.length()-1).equals("?");
	}
	
	/**
	 * Checks if the sentence is inferred or asserted.
	 * 
	 * @return true if the sentence is inferred, false if it is asserted.
	 */
	public boolean isInferred() {
		return owner == null;
	}
	
	/**
	 * Checks whether the sentence contains the given word form (by word number) of the
	 * given ontology element.
	 * 
	 * @param e The ontology element.
	 * @param wordNumber The word number.
	 * @return true if the word form occurs in this sentence.
	 */
	public boolean contains(OntologyElement e, int wordNumber) {
		if (textContainer == null) {
			tokenize();
		}
		for (TextElement t : textContainer.getTextElements()) {
			if (t instanceof OntologyTextElement) {
				OntologyTextElement ot = (OntologyTextElement) t;
				if (e == ot.getOntologyElement() && wordNumber == -1) return true;
				if (e == ot.getOntologyElement() && wordNumber == ot.getWordNumber()) return true;
			}
		}
		return false;
	}

	/**
	 * Checks whether the sentence contains the given ontology element (no matter which
	 * word form).
	 * 
	 * @param e The ontology element.
	 * @return true if the ontology element occurs in this sentence.
	 */
	public boolean contains(OntologyElement e) {
		return contains(e, -1);
	}
	
	/**
	 * Returns all individuals that answer this question. Questions in AceWiki are "DL Queries".
	 * They describe a concept and the answer consists of all individuals that belong to this concept.
	 * The null value is returned if the sentence is not a question.
	 * 
	 * @return A list of individuals that are the answer for the question.
	 * @see Ontology#getAnswer(Sentence)
	 */
	public List<Individual> getAnswer() {
		if (!isQuestion()) return null;
		
		Ontology o = getOntology();
		if (answerCacheStateID != o.getStateID()) {
			answerCache = o.getAnswer(this);
			answerCacheStateID = o.getStateID();
		}
		return new ArrayList<Individual>(answerCache);
	}
	
	/**
	 * Returns true if the sentence is a question and the answer to the question is cached and does
	 * not have to be recalculated.
	 * 
	 * @return true if the answer is cached.
	 */
	public boolean isAnswerCached() {
		if (!isQuestion()) return false;
		return answerCacheStateID == getOntology().getStateID();
	}
	
	private static String getUnderscoredText(TextContainer textContainer) {
		String t = "";
		for (TextElement te : textContainer.getTextElements()) {
			if (te instanceof OntologyTextElement) {
				t += " " + ((OntologyTextElement) te).getUnderscoredText();
			} else if (te.getText().matches("[.?]")) {
				t += te.getText();
			} else {
				t += " " + te.getText();
			}
		}
		if (t.length() > 0) {
			t = t.substring(1);
		}
		return t;
	}
	
	String serialize() {
		if (textContainer == null) {
			tokenize();
		}
		String s;
		if (integrated) {
			s = "|";
		} else {
			s = "#";
		}
		for (TextElement te : textContainer.getTextElements()) {
			if (te instanceof OntologyTextElement) {
				OntologyTextElement ot = (OntologyTextElement) te;
				s += " <" + ot.getOntologyElement().getId() + "," + ot.getWordNumber() + ">";
			} else {
				s += " " + te.getText();
			}
		}
		return s + "\n";
	}
	
	public String toString() {
		return getText();
	}

}
