// This file is part of the Attempto Java Packages.
// Copyright 2008, Attempto Group, University of Zurich (see http://attempto.ifi.uzh.ch).
//
// The Attempto Java Packages is free software: you can redistribute it and/or modify it under the
// terms of the GNU Lesser General Public License as published by the Free Software Foundation,
// either version 3 of the License, or (at your option) any later version.
//
// The Attempto Java Packages is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public License along with the Attempto
// Java Packages. If not, see http://www.gnu.org/licenses/.

package ch.uzh.ifi.attempto.acewiki.core.text;

import ch.uzh.ifi.attempto.acewiki.core.ontology.Individual;
import ch.uzh.ifi.attempto.acewiki.core.ontology.NounConcept;
import ch.uzh.ifi.attempto.acewiki.core.ontology.OfRole;
import ch.uzh.ifi.attempto.acewiki.core.ontology.OntologyElement;
import ch.uzh.ifi.attempto.acewiki.core.ontology.TrAdjRole;
import ch.uzh.ifi.attempto.acewiki.core.ontology.VerbRole;
import ch.uzh.ifi.attempto.preditor.text.TextElement;

/**
 * This factory class creates text element objects on the basis of ontology elements.
 * 
 * @author Tobias Kuhn
 */
public class TextElemFactory {
	
	private TextElemFactory() {}  // no instances allowed
	
	/**
	 * Creates a new text element for the given word form (by word form id) of the given
	 * ontology element.
	 * 
	 * @param el The ontology element.
	 * @param wordNumber The word form id.
	 * @return The newly created text element.
	 */
	public static OntologyTextElement createTextElement(OntologyElement el, int wordNumber) {
		if (el instanceof NounConcept) {
			if (wordNumber == 0) {
				return new NounTextElem((NounConcept) el);
			} else if (wordNumber == 1) {
				return new OntoTextElem(el, 1, "NOUNPL");
			}
		} else if (el instanceof Individual) {
			if (wordNumber == 0) return new OntoTextElem(el, 0, "PROPERNAME");
		} else if (el instanceof VerbRole) {
			if (wordNumber == 0) {
				return new OntoTextElem(el, 0, "VERB");
			} else if (wordNumber == 1) {
				return new OntoTextElem(el, 1, "VERBINF");
			} else if (wordNumber == 2) {
				return new OntoTextElem(el, 2, "PVERB");
			}
		} else if (el instanceof OfRole) {
			if (wordNumber == 0) return new OntoTextElem(el, 0, "NOUNOF");
		} else if (el instanceof TrAdjRole) {
			if (wordNumber == 0) return new OntoTextElem(el, 0, "TRADJ");
		}
		return null;
	}
	
	/**
	 * Creates a new text element for the default word form of the given ontology element.
	 * 
	 * @param el The ontology element.
	 * @return The newly created text element.
	 */
	public static TextElement createTextElement(OntologyElement el) {
		return createTextElement(el, 0);
	}

}
