// This file is part of the Attempto Java Packages.
// Copyright 2008, Attempto Group, University of Zurich (see http://attempto.ifi.uzh.ch).
//
// The Attempto Java Packages is free software: you can redistribute it and/or modify it under the
// terms of the GNU Lesser General Public License as published by the Free Software Foundation,
// either version 3 of the License, or (at your option) any later version.
//
// The Attempto Java Packages is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public License along with the Attempto
// Java Packages. If not, see http://www.gnu.org/licenses/.

package ch.uzh.ifi.attempto.acewiki.gui.editor;

import nextapp.echo2.app.ApplicationInstance;
import nextapp.echo2.app.CheckBox;
import nextapp.echo2.app.ResourceImageReference;
import nextapp.echo2.app.event.ActionListener;
import ch.uzh.ifi.attempto.acewiki.Wiki;
import ch.uzh.ifi.attempto.acewiki.core.ontology.Individual;
import ch.uzh.ifi.attempto.acewiki.core.ontology.OntologyElement;
import ch.uzh.ifi.attempto.ape.FunctionWords;
import ch.uzh.ifi.attempto.echocomp.TextField;
import ch.uzh.ifi.attempto.echocomp.WindowPane;

/**
 * This class represents a form to create or modify proper names.
 * 
 * @author Tobias Kuhn
 */
public class ProperNameForm extends FormPane {
	
	private static final long serialVersionUID = 7860047859937196093L;
	
	private TextField nameField = new TextField();
	private CheckBox defArticleCheckBox = new CheckBox();
	
	private Individual ind;
	
	/**
	 * Creates a new proper name form.
	 * 
	 * @param ind The individual that is represented by the proper name.
	 * @param window The host window of the form.
	 * @param wiki The wiki instance.
	 * @param actionListener The actionlistener.
	 */
	public ProperNameForm(Individual ind, WindowPane window, Wiki wiki, ActionListener actionListener) {
		super(window, wiki, actionListener);
		this.ind = ind;
		
		setIconRow("individual",
				"Every proper name represents a certain individual. " +
				"The country \"Switzerland\", the person \"Bob Dylan\", the river \"Nile\", " +
				"and the organization \"United Nations\" are typical examples for such individuals. " +
				"Some proper names require a definite article (\"the Nile\", \"the United Nations\") " +
				"and others do not (\"Switzerland\", \"Bob Dylan\")."
			);
		addRow("proper name", nameField, "examples: Switzerland, Bob Dylan, Nile, United Nations", true);
		addRow("definite article", defArticleCheckBox, "examples: the Nile, the United Nations", false);
		
		nameField.setText(ind.getPrettyWord(1));
		defArticleCheckBox.setSelected(ind.hasDefiniteArticle());
		defArticleCheckBox.setStateIcon(new ResourceImageReference("ch/uzh/ifi/attempto/acewiki/gui/img/notchecked.png"));
		defArticleCheckBox.setSelectedStateIcon(new ResourceImageReference("ch/uzh/ifi/attempto/acewiki/gui/img/checked.png"));
		
		ApplicationInstance.getActive().setFocusedComponent(nameField);
	}
	
	/**
	 * Creates a new creator window for proper names.
	 * 
	 * @param wiki The wiki instance.
	 * @param actionListener The actionlistener.
	 * @return The new creator window.
	 */
	public static WordEditorWindow createCreatorWindow(Wiki wiki, ActionListener actionListener) {
		WordEditorWindow creatorWindow = WordEditorWindow.createCreatorWindow();
		creatorWindow.addTab(new ProperNameForm(new Individual(), creatorWindow, wiki, actionListener));
		return creatorWindow;
	}
	
	/**
	 * Creates a new editor window for proper names.
	 * 
	 * @param ind The individual that is represented by the proper name that should be edited.
	 * @param wiki The wiki instance.
	 * @return The new editor window.
	 */
	public static WordEditorWindow createEditorWindow(Individual ind, Wiki wiki) {
		WordEditorWindow editorWindow = WordEditorWindow.createEditorWindow();
		editorWindow.addTab(new ProperNameForm(ind, editorWindow, wiki, wiki));
		return editorWindow;
	}

	protected void save() {
		Wiki wiki = getWiki();
		String name = normalize(nameField.getText());
		if (name.toLowerCase().startsWith("the_")) {
			name = name.substring(4);
		}
		String nameP = name.replace("_", " ");
		
		if (name.equals("")) {
			wiki.log("edit", "error: no word defined");
			showErrorMessage("No proper name defined: Please specify a name.");
			return;
		}
		if (!isValidString(name)) {
			wiki.log("edit", "error: word contains invalid character");
			showErrorMessage("Invalid character: Only a-z, A-Z, 0-9, -, and spaces are allowed, " +
				"and the first character must be one of a-z A-Z.");
			return;
		}
		if (FunctionWords.isFunctionWord(name)) {
			wiki.log("edit", "error: word is predefined");
			showErrorMessage("'" + nameP + "' is a predefined word and cannot be used here.");
			return;
		}
		OntologyElement oe = wiki.getOntology().get(name);
		if (oe != null && oe != ind) {
			wiki.log("edit", "error: word is already used");
			showErrorMessage("The word '" + nameP + "' is already used. Please use a different one.");
			return;
		}
		String word = name;
		if (defArticleCheckBox.isSelected()) {
			word = "the " + name;
		}
		ind.setWords(word, name);
		wiki.log("edit", "proper name: " + word);
		if (ind.getOntology() == null) {
			ind.registerAt(getWiki().getOntology());
		}
		finished(ind);
	}
	
	public String toString() {
		return "Proper Name";
	}

}
