// This file is part of the Attempto Java Packages.
// Copyright 2008, Attempto Group, University of Zurich (see http://attempto.ifi.uzh.ch).
//
// The Attempto Java Packages is free software: you can redistribute it and/or modify it under the
// terms of the GNU Lesser General Public License as published by the Free Software Foundation,
// either version 3 of the License, or (at your option) any later version.
//
// The Attempto Java Packages is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public License along with the Attempto
// Java Packages. If not, see http://www.gnu.org/licenses/.

package ch.uzh.ifi.attempto.chartparser;

import java.util.HashMap;
import java.util.List;

/**
 * This class represents a grammar rule. A rule consists of a nonterminal category as its head,
 * a sequence of categories as its body, and a boolean value that describes whether it is accessible
 * or not. If a rule is not accessible then this means that no category of the body and no subcategory
 * of those categories is accessible by any category outside of the body. This can be used to define
 * which parts of the sentence are accessible for anaphoric references from a given position.
 * 
 * @author Tobias Kuhn
 */
public class Rule {
	
	private final Nonterminal head;
	private final Category[] body;
	private final boolean accessible;
	
	/**
	 * Creates a new rule.
	 * 
	 * @param head The head category.
	 * @param accessible Defines whether the rule is accessible or not.
	 * @param body The body category sequence.
	 */
	public Rule(Nonterminal head, boolean accessible, Category... body) {
		this.head = head;
		this.body = body;
		this.accessible = accessible;
	}
	
	/**
	 * Creates a new rule that is accessible.
	 * 
	 * @param head The head category.
	 * @param body The body category sequence.
	 */
	public Rule(Nonterminal head, Category... body) {
		this(head, true, body);
	}
	
	/**
	 * Creates a new rule.
	 * 
	 * @param categories The first category of this list stands for the head category (it has to be
	 *     a Nonterminal object). The rest stands for the body categories.
	 * @param accessible Defines whether the rule is accessible or not.
	 */
	public Rule(List<Category> categories, boolean accessible) {
		this.accessible = accessible;
		this.head = (Nonterminal) categories.get(0);
		categories.remove(0);
		this.body = categories.toArray(new Category[0]);
	}
	
	/**
	 * Creates a new rule that is accessible.
	 * 
	 * @param categories The first category of this list stands for the head category (it has to be
	 *     a Nonterminal object). The rest stands for the body categories.
	 */
	public Rule(List<Category> categories) {
		this(categories, true);
	}
	
	/**
	 * Returns the head category of this rule.
	 * 
	 * @return The head category.
	 */
	public Nonterminal getHead() {
		return head;
	}
	
	/**
	 * Returns the body category sequence of this rule.
	 * 
	 * @return The body category sequence.
	 */
	public Category[] getBody() {
		return body;
	}
	
	/**
	 * Returns true if the body is empty (i.e. it is an epsilon-rule).
	 * 
	 * @return true if the body is empty.
	 */
	public boolean hasEmptyBody() {
		return body.length == 0;
	}
	
	/**
	 * Returns the first category of the body.
	 * 
	 * @return The first category of the body.
	 */
	public Category getFirst() {
		return body[0];
	}
	
	/**
	 * Returns true if the rule is accessible.
	 * 
	 * @return true if the rule is accessible.
	 */
	public boolean isAccessible() {
		return accessible;
	}
	
	/**
	 * Creates an edge on the basis of this rule. Such edges are used by the chart parser.
	 * 
	 * @param pos The position of the edge to be created.
	 * @return The edge.
	 */
	Edge createEdge(int pos) {
		return new Edge(pos, pos, getHead(), getBody(), accessible);
	}
	
	/**
	 * Creates a deep copy of this rule.
	 * 
	 * @return A deep copy.
	 */
	public Rule deepCopy() {
		return deepCopy(new HashMap<Integer, StringEntity>());
	}
	
	/**
	 * Creates a deep copy of this rule using the given string entities. This method is
	 * usually called form another deepCopy-method.
	 * 
	 * @param entities The string entities to be used.
	 * @return A deep copy.
	 */
	public Rule deepCopy(HashMap<Integer, StringEntity> entities) {
		Nonterminal headC = (Nonterminal) head.deepCopy(entities);
		Category[] bodyC = new Category[body.length];
		for (int i=0; i < body.length ; i++) {
			bodyC[i] = body[i].deepCopy(entities);
		}
		return new Rule(headC, accessible, bodyC);
	}
	
	public String toString() {
		String s = head + " ";
		if (accessible) {
			s += "=>";
		} else {
			s += "~>";
		}
		for (Category c : body) {
			s += " " + c;
		}
		return s;
	}

}
