// This file is part of the Attempto Java Packages.
// Copyright 2008, Attempto Group, University of Zurich (see http://attempto.ifi.uzh.ch).
//
// The Attempto Java Packages is free software: you can redistribute it and/or modify it under the
// terms of the GNU Lesser General Public License as published by the Free Software Foundation,
// either version 3 of the License, or (at your option) any later version.
//
// The Attempto Java Packages is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public License along with the Attempto
// Java Packages. If not, see http://www.gnu.org/licenses/.

package ch.uzh.ifi.attempto.echocomp;

import java.util.HashMap;

import nextapp.echo2.app.ApplicationInstance;
import nextapp.echo2.app.Column;
import nextapp.echo2.app.Component;
import nextapp.echo2.app.TaskQueueHandle;

/**
 * This abstract class can be used to create components that are initialized asynchronously in the background.
 * This is makes sense for components that require some time to create themselves (e.g. because of time
 * expensive calculations).
 * 
 * @author Tobias Kuhn
 */
public abstract class DelayedComponent extends Column {
	
	private static HashMap<ApplicationInstance, TaskQueueHandle> taskQueues = new HashMap<ApplicationInstance, TaskQueueHandle>();
	
	/**
	 * Creates a new delayed component which shows the temporary component until the real component is
	 * ready.
	 * 
	 * @param tempComponent The temporary component.
	 */
	public DelayedComponent(Component tempComponent) {
		if (tempComponent != null) {
			add(tempComponent);
		}
		
		final ApplicationInstance application = ApplicationInstance.getActive();
		TaskQueueHandle taskQueueTemp = taskQueues.get(application);
		if (taskQueueTemp == null) {
			taskQueueTemp = application.createTaskQueue();
			taskQueues.put(application, taskQueueTemp);
		}
		final TaskQueueHandle taskQueue = taskQueueTemp;
		
		Thread thread = new Thread() {
			public synchronized void run() {
				final Component c = initComponent();
				application.enqueueTask(
					taskQueue,
					new Runnable() {
						public synchronized void run() {
							DelayedComponent.this.removeAll();
							DelayedComponent.this.add(c);
						}
					}
				);
			}
		};
		
		thread.start();
	}
	
	
	/**
	 * Creates a new delayed component with no temporary component.
	 */
	public DelayedComponent() {
		this(null);
	}
	
	/**
	 * This method should contain the (possibly time-consuming) operations to create the actual GUI
	 * component. This operation will be performed asynchronously. As soon as it is finished, the
	 * temporary component (if present) is replaced by the component this method returns.
	 * 
	 * @return The GUI component.
	 */
	public abstract Component initComponent();

}
