// This file is part of the Attempto Java Packages.
// Copyright 2008, Attempto Group, University of Zurich (see http://attempto.ifi.uzh.ch).
//
// The Attempto Java Packages is free software: you can redistribute it and/or modify it under the
// terms of the GNU Lesser General Public License as published by the Free Software Foundation,
// either version 3 of the License, or (at your option) any later version.
//
// The Attempto Java Packages is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public License along with the Attempto
// Java Packages. If not, see http://www.gnu.org/licenses/.

package ch.uzh.ifi.attempto.preditor;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import ch.uzh.ifi.attempto.chartparser.ChartParser;
import ch.uzh.ifi.attempto.chartparser.Terminal;
import ch.uzh.ifi.attempto.preditor.text.RefTextElement;
import ch.uzh.ifi.attempto.preditor.text.RefableTextElement;
import ch.uzh.ifi.attempto.preditor.text.TextContainer;
import ch.uzh.ifi.attempto.preditor.text.TextElement;
import ch.uzh.ifi.attempto.preditor.text.VarTextElement;

/**
 * This abstract class is used for the predictive editor to create menus on the basis of categories of the grammar.
 *  
 * @author Tobias Kuhn
 */
public abstract class MenuCreator {
	
	private HashMap<String, MenuBlockContent> contentsMap = new HashMap<String, MenuBlockContent>();
	private List<MenuBlockContent> contentsList = new ArrayList<MenuBlockContent>();
	private ChartParser parser;
	private TextContainer textContainer;
	
	/**
	 * Initializes a new menu creator object.
	 */
	public MenuCreator() {
	}
	
	/**
	 * An implementation of this abstract method should process the category by calling {@link #addMenuItem addMenuItem}
	 * and/or {@link #addMenuEntry addMenuEntry}.
	 * 
	 * @param category The category of the grammar to be processed.
	 */
	public abstract void processCategory(Terminal category);
	
	/**
	 * This abstract method is called when the predictive editor is refreshed (before the first call of
	 * {@link #processCategory processCategory}). This is a good place for calling {@link #prepareMenuBlock prepareMenuBlock}.
	 */
	public abstract void initMenuCreation();
	
	/**
	 * Prepares a new menu block (if there is no such menu block already with the same name). The menu blocks are
	 * shown in the predictive editor in the same order as they have been prepared.
	 * 
	 * @param menuBlockName The name of the menu block.
	 */
	public void prepareMenuBlock(String menuBlockName) {
		prepareMenuBlock(menuBlockName, false);
	}

	/**
	 * Prepares a new menu block (if there is no such menu block already with the same name). The second parameter
	 * defines whether the items of the menu block should be sorted. The menu blocks are
	 * shown in the predictive editor in the same order as they have been prepared.
	 * 
	 * @param menuBlockName The name of the menu block.
	 * @param doSort true if the items should be sorted.
	 */
	public void prepareMenuBlock(String menuBlockName, boolean doSort) {
		if (contentsMap.containsKey(menuBlockName)) {
			return;
		}
		MenuBlockContent menuBlockContent = new MenuBlockContent(menuBlockName, doSort);
		contentsMap.put(menuBlockName, menuBlockContent);
		contentsList.add(menuBlockContent);
	}
	
	/**
	 * Adds the menu item to the given menu block. If no menu block of this name has been prepared at this point then
	 * the preparation is done first.
	 * 
	 * @param menuBlockName The name of the menu block.
	 * @param menuItem The menu item to be added to the menu block.
	 */
	public void addMenuItem(String menuBlockName, MenuItem menuItem) {
		if (!contentsMap.containsKey(menuBlockName)) {
			prepareMenuBlock(menuBlockName);
		}
		contentsMap.get(menuBlockName).addItem(menuItem);
	}
	
	/**
	 * Adds a new menu entry containing the text element to the menu block. If no menu block of this name has been
	 * prepared at this point then the preparation is done first.
	 * 
	 * @param menuBlockName The name of the menu block.
	 * @param textElement The text element of the menu entry that is created and then added to the menu block.
	 */
	public void addMenuEntry(String menuBlockName, TextElement textElement) {
		if (!contentsMap.containsKey(menuBlockName)) {
			prepareMenuBlock(menuBlockName);
		}
		contentsMap.get(menuBlockName).addEntry(textElement);
	}
	
	/**
	 * This method looks for referenceable text elements earlier in the token list, given the category of the next
	 * token. For each of those text elements, a new reference text element is created and added to the menu block.
	 * 
	 * @param menuBlockName The name of the menu block.
	 * @param category The category name for which the previous tokens should be checked for referenceability.
	 * @see #getAccessibleTextElements(String)
	 * @see RefableTextElement
	 * @see RefTextElement
	 */
	public void addReferenceEntries(String menuBlockName, String category) {
		ArrayList<String> refStrings = new ArrayList<String>();
		for (TextElement el : getAccessibleTextElements(category)) {
			if (!(el instanceof RefableTextElement)) continue;
			
			RefableTextElement refEl = (RefableTextElement) el;
			boolean isOverridden = false;
			String refText = refEl.getReferenceText();
			for (String s : refStrings) {
				if (s.equals(refText) || s.startsWith(refText + " ")) {
					isOverridden = true;
					break;
				}
			}
			if (!isOverridden) {
				refStrings.add(refText);
				addMenuEntry(menuBlockName, new RefTextElement(refEl, category));
			}
		}
	}
	
	/**
	 * This method adds text elements that contain variables (in the form of "X", "Y", "Z", "X1", "Y1", "Z1", "X2", and so on)
	 * to the menu block.
	 * 
	 * @param menuBlockName The name of the menu block.
	 * @param num Variables are created up to this number.
	 * @param category The category name of the variable tokens.
	 * @see VarTextElement
	 */
	public void addVariableEntries(String menuBlockName, int num, String category) {
		ArrayList<String> varNames = new ArrayList<String>();
		varNames.add("X");
		varNames.add("Y");
		varNames.add("Z");
		for (int i=1; i<=num; i++) {
			varNames.add("X" + i);
			varNames.add("Y" + i);
			varNames.add("Z" + i);
		}
		for (TextElement el : textContainer.getTextElements()) {
			if (!(el instanceof VarTextElement)) continue;
			varNames.remove(((VarTextElement) el).getText());
		}
		for (String s : varNames) {
			addMenuEntry(menuBlockName, new VarTextElement(s, category));
		}
	}
	
	/**
	 * This method returns all text elements that are accessible for the given next token.
	 * 
	 * @param nextToken The next token.
	 * @return A list of all accessible text elements.
	 */
	public List<TextElement> getAccessibleTextElements(String nextToken) {
		List<TextElement> elements = new ArrayList<TextElement>();
		
		boolean[] pos = parser.getAccessiblePositions(nextToken);
		for (int i=pos.length-1 ; i >= 0 ; i--) {
			if (!pos[i]) continue;
			TextElement el = textContainer.getTextElements().get(i);
			elements.add(el);
		}
		
		return elements;
	}
	
	List<MenuBlockContent> createMenu(ChartParser parser, TextContainer textContainer) {
		this.parser = parser;
		this.textContainer = textContainer;
		contentsMap.clear();
		contentsList.clear();
		
		initMenuCreation();
		
		for (Terminal t : parser.nextTokens()) {
			processCategory(t);
		}
		
		return new ArrayList<MenuBlockContent>(contentsList);
	}

}
