// This file is part of the Attempto Java Packages.
// Copyright 2008, Attempto Group, University of Zurich (see http://attempto.ifi.uzh.ch).
//
// The Attempto Java Packages is free software: you can redistribute it and/or modify it under the
// terms of the GNU Lesser General Public License as published by the Free Software Foundation,
// either version 3 of the License, or (at your option) any later version.
//
// The Attempto Java Packages is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public License along with the Attempto
// Java Packages. If not, see http://www.gnu.org/licenses/.

package ch.uzh.ifi.attempto.acewiki.gui.editor;

import nextapp.echo2.app.Alignment;
import nextapp.echo2.app.Column;
import nextapp.echo2.app.Component;
import nextapp.echo2.app.ContentPane;
import nextapp.echo2.app.Extent;
import nextapp.echo2.app.Font;
import nextapp.echo2.app.Grid;
import nextapp.echo2.app.Insets;
import nextapp.echo2.app.ResourceImageReference;
import nextapp.echo2.app.Row;
import nextapp.echo2.app.event.ActionEvent;
import nextapp.echo2.app.event.ActionListener;
import nextapp.echo2.app.layout.GridLayoutData;
import ch.uzh.ifi.attempto.acewiki.Wiki;
import ch.uzh.ifi.attempto.acewiki.core.ontology.OntologyElement;
import ch.uzh.ifi.attempto.acewiki.core.text.OntologyTextElement;
import ch.uzh.ifi.attempto.acewiki.core.text.TextElemFactory;
import ch.uzh.ifi.attempto.echocomp.GeneralButton;
import ch.uzh.ifi.attempto.echocomp.Label;
import ch.uzh.ifi.attempto.echocomp.MessageWindow;
import ch.uzh.ifi.attempto.echocomp.SolidLabel;
import ch.uzh.ifi.attempto.echocomp.TextField;
import ch.uzh.ifi.attempto.echocomp.WindowPane;

/**
 * This abstract class contains the basic structure for forms to create and modify words
 * (represented by ontology elements).
 * 
 * @author Tobias Kuhn
 */
public abstract class FormPane extends ContentPane implements ActionListener {

	private Wiki wiki;
	private ActionListener actionListener;
	
	private Column column;
	private WindowPane window;
	private Grid iconRow;
	private GeneralButton okButton;
	private GeneralButton cancelButton;
	
	/**
	 * Initializes the form pane.
	 * 
	 * @param window The host window.
	 * @param wiki The wiki instance.
	 * @param actionListener The actionlistener.
	 */
	protected FormPane(WindowPane window, Wiki wiki, ActionListener actionListener) {
		this.window = window;
		this.wiki = wiki;
		this.actionListener = actionListener;
		
		Grid grid = new Grid(1);
		grid.setRowHeight(0, new Extent(330));
		grid.setRowHeight(1, new Extent(30));
		
		column = new Column();
		column.setInsets(new Insets(10, 20, 0, 0));
		column.setCellSpacing(new Extent(10));
		GridLayoutData gridLayout = new GridLayoutData();
		gridLayout.setAlignment(new Alignment(Alignment.LEFT, Alignment.TOP));
		column.setLayoutData(gridLayout);
		grid.add(column);
		
		iconRow = new Grid(2);
		iconRow.setRowHeight(0, new Extent(110));
		iconRow.setColumnWidth(0, new Extent(100));
		iconRow.setInsets(new Insets(0, 0, 10, 20));
		column.add(iconRow);
		
		Row footerRow = new Row();
		footerRow.setInsets(new Insets(10, 0, 0, 0));
		footerRow.add(new Label("* required field", Font.ITALIC, 11));
		grid.add(footerRow);
		
		Row buttonBar = new Row();
		buttonBar.setAlignment(new Alignment(Alignment.RIGHT, Alignment.CENTER));
		buttonBar.setInsets(new Insets(10, 10, 10, 10));
		buttonBar.setCellSpacing(new Extent(5));
		okButton = new GeneralButton("OK", 70, this);
		buttonBar.add(okButton);
		cancelButton = new GeneralButton("Cancel", 70, this);
		buttonBar.add(cancelButton);
		grid.add(buttonBar);
		
		add(grid);
	}
	
	/**
	 * Returns the wiki instance.
	 * 
	 * @return The wiki instance.
	 */
	protected Wiki getWiki() {
		return wiki;
	}
	
	/**
	 * This method should try to save the word with the current properties and should show
	 * error messages if this is not successful. In the case of success, one of the
	 * finished-methods has to be called.
	 * 
	 * @see #finished(OntologyElement)
	 * @see #finished(OntologyElement, int)
	 */
	protected abstract void save();
	
	/**
	 * This method should be called when the saving process is finished successfully.
	 * 
	 * @param el The created or modified ontology element.
	 */
	protected void finished(OntologyElement el) {
		finished(el, 0);
	}
	
	/**
	 * This method should be called when the saving process is finished successfully.
	 * 
	 * @param el The created or modified ontology element.
	 * @param wordNumber The word form id.
	 */
	protected void finished(OntologyElement el, int wordNumber) {
		window.setVisible(false);
		dispose();
		
		// a text element is used to store the ontology element and the word number in one object:
		OntologyTextElement te = TextElemFactory.createTextElement(el, wordNumber);
		actionListener.actionPerformed(new ActionEvent(te, ""));
	}
	
	/**
	 * Adds a new row to the form.
	 * 
	 * @param labelText The text for the label shown on the left hand side of the component.
	 * @param component The component, i.e. a text field.
	 * @param explanation An explanation text shown under the component.
	 * @param required Defines whether the component should be marked as required.
	 */
	protected void addRow(String labelText, Component component, String explanation, boolean required) {
		Grid grid = new Grid(3);
		grid.setInsets(new Insets(0, 0, 5, 0));
		grid.setColumnWidth(0, new Extent(140));
		grid.add(new SolidLabel(labelText, Font.ITALIC));
		if (component instanceof TextField) {
			((TextField) component).setWidth(new Extent(530));
		}
		grid.add(component);
		if (required) {
			grid.add(new Label("*", Font.ITALIC, 11));
		} else {
			grid.add(new Label());
		}
		grid.add(new Label());
		grid.add(new Label(explanation, Font.ITALIC, 11));
		column.add(grid);
	}
	
	/**
	 * Sets the icon row. The icon row is shown at the top of the form and shows an icon of the
	 * ontological structure that is represented by the word. On the right hand side of the icon,
	 * an explanatory text is shown.
	 * 
	 * @param iconName The name of the icon, one of "individual", "concept", or "role".
	 * @param text The explanatory text.
	 */
	protected void setIconRow(String iconName, String text) {
		iconRow.removeAll();
		iconRow.add(new Label(new ResourceImageReference("ch/uzh/ifi/attempto/acewiki/gui/img/" + iconName + ".png")));
		iconRow.add(new Label(text, Font.ITALIC));
	}
	
	/**
	 * Shows an error message.
	 * 
	 * @param text The error text.
	 */
	protected void showErrorMessage(String text) {
		wiki.showWindow(new MessageWindow("Error", text, window, "OK"));
	}
	
	public void actionPerformed(ActionEvent e) {
		if (e.getSource() == cancelButton) {
			window.setVisible(false);
			dispose();
		} else {
			save();
		}
	}
	
	/**
	 * Normalizes the string. All white spaces are replaced by underscores. Afterwards, leading and
	 * trailing underscores are removed and successive underscores are replaced by just one underscore.
	 * 
	 * @param s The string to be normalized.
	 * @return The normalized string.
	 */
	protected static String normalize(String s) {
		return s.replaceAll("\\s", "_").replaceAll("(_)+", "_").replaceAll("^_", "").replaceAll("_$", "");
	}
	
	/**
	 * Returns true if the string is a valid to be used as a word form. The first character must be
	 * one of a-z, A-Z. The following characters have to be one of a-z, A-Z, 0-9, -, _.
	 * 
	 * @param s The string to be checked for validity.
	 * @return true if the string is valid.
	 */
	protected static boolean isValidString(String s) {
		if (s.length() == 0) return true;
		boolean first = true;
		for (byte b : s.getBytes()) {
			if (first) {
				first = false;
				if (b >= 'a' && b <= 'z') continue;
				if (b >= 'A' && b <= 'Z') continue;
				return false;
			} else {
				if (b >= 'a' && b <= 'z') continue;
				if (b >= 'A' && b <= 'Z') continue;
				if (b >= '0' && b <= '9') continue;
				if (b == '-') continue;
				if (b == '_') continue;
				return false;
			}
		}
		return true;
	}

}
