// This file is part of the Attempto Java Packages.
// Copyright 2008, Attempto Group, University of Zurich (see http://attempto.ifi.uzh.ch).
//
// The Attempto Java Packages is free software: you can redistribute it and/or modify it under the
// terms of the GNU Lesser General Public License as published by the Free Software Foundation,
// either version 3 of the License, or (at your option) any later version.
//
// The Attempto Java Packages is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public License along with the Attempto
// Java Packages. If not, see http://www.gnu.org/licenses/.

package ch.uzh.ifi.attempto.acewiki.gui.editor;

import java.util.ArrayList;

import nextapp.echo2.app.Alignment;
import nextapp.echo2.app.ApplicationInstance;
import nextapp.echo2.app.Border;
import nextapp.echo2.app.Color;
import nextapp.echo2.app.Component;
import nextapp.echo2.app.ContentPane;
import nextapp.echo2.app.Extent;
import nextapp.echo2.app.Font;
import nextapp.echo2.app.Insets;
import nextapp.echo2.app.event.ChangeEvent;
import nextapp.echo2.app.event.ChangeListener;
import ch.uzh.ifi.attempto.echocomp.Style;
import ch.uzh.ifi.attempto.echocomp.TextField;
import ch.uzh.ifi.attempto.echocomp.WindowPane;
import echopointng.ButtonEx;
import echopointng.TabbedPane;
import echopointng.tabbedpane.DefaultTabModel;

/**
 * This class represents the word editor that is used to create or modify words. It can contain
 * several tabs for several word forms.
 * 
 * @author Tobias Kuhn
 */
public class WordEditorWindow extends WindowPane implements ChangeListener {
	
	private static final long serialVersionUID = 6805275173727379038L;
	
	private TabbedPane tabbedPane = new TabbedPane();
	private ArrayList<ContentPane> tabs = new ArrayList<ContentPane>();
	
	private WordEditorWindow(String title) {
		setModal(true);
		setTitle(title);
		setTitleFont(new Font(Style.fontTypeface, Font.ITALIC, new Extent(13)));
		setWidth(new Extent(753));
		setHeight(new Extent(503));
		setResizable(false);
		setTitleBackground(Style.windowTitleBackground);
		setStyleName("Default");
		
		tabbedPane.setOutsets(new Insets(10));
		tabbedPane.setTabPlacement(Alignment.TOP);
		tabbedPane.setBorder(new Border(1, Color.BLACK, Border.STYLE_INSET));
		tabbedPane.setHeight(new Extent(400));
		tabbedPane.getSelectionModel().addChangeListener(this);
		
		add(tabbedPane);
	}
	
	/**
	 * Creates a new word editor window.
	 * 
	 * @return The new word editor window.
	 */
	public static WordEditorWindow createEditorWindow() {
		return new WordEditorWindow("Word Editor");
	}
	
	/**
	 * Creates a new word creator window.
	 * 
	 * @return The new word creator window.
	 */
	public static WordEditorWindow createCreatorWindow() {
		return new WordEditorWindow("Word Creator");
	}
	
	/**
	 * Adds a new tab to the window. The toString() value is used as the title of the tab.
	 * 
	 * @param pane The pane that should be shown in the new tab.
	 */
	public void addTab(ContentPane pane) {
		DefaultTabModel tabModel = (DefaultTabModel) tabbedPane.getModel();
		ButtonEx tab = new ButtonEx(pane.toString(), null);
		tab.setStyle(DefaultTabModel.DEFAULT_TOP_ALIGNED_STYLE);
		tab.setFont(new Font(Style.fontTypeface, Font.ITALIC, new Extent(13)));
		tab.setInsets(new Insets(5, 2));
		tabModel.addTab(tab, pane);
		tabs.add(pane);
		doFocus();
	}
	
	private void doFocus() {
		int i = tabbedPane.getSelectedIndex();
		if (i < tabs.size()) {
			doFocus(tabs.get(i));
		}
	}
	
	private boolean doFocus(Component c) {
		if (c instanceof TextField) {
			ApplicationInstance.getActive().setFocusedComponent((TextField) c);
			return true;
		} else {
			for (Component child : c.getComponents()) {
				boolean b = doFocus(child);
				if (b) return true;
			}
		}
		return false;
	}
	
	public void stateChanged(ChangeEvent e) {
		doFocus();
	}
	
}
