// This file is part of the Attempto Java Packages.
// Copyright 2008, Attempto Group, University of Zurich (see http://attempto.ifi.uzh.ch).
//
// The Attempto Java Packages is free software: you can redistribute it and/or modify it under the
// terms of the GNU Lesser General Public License as published by the Free Software Foundation,
// either version 3 of the License, or (at your option) any later version.
//
// The Attempto Java Packages is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public License along with the Attempto
// Java Packages. If not, see http://www.gnu.org/licenses/.

package ch.uzh.ifi.attempto.chartparser;

import java.util.HashMap;
import java.util.Set;

/**
 * This class represents a grammatical category.
 * 
 * @author Tobias Kuhn
 */
public abstract class Category {
	
	private static int skNumber = 0;
	
	private final String name;
	private HashMap<String, StringRef> features = new HashMap<String, StringRef>();
	
	/**
	 * Creates a new category.
	 * 
	 * @param name The name of the category.
	 */
	public Category(String name) {
		this.name = name;
	}
	
	/**
	 * Returns the name of the category.
	 * 
	 * @return The name of the category.
	 */
	public String getName() {
		return name;
	}
	
	/**
	 * Sets a feature.
	 * 
	 * @param featureName The feature name
	 * @param featureValue
	 */
	public void setFeature(String featureName, StringRef featureValue) {
		features.put(featureName, featureValue);
	}
	
	/**
	 * Returns a feature value.
	 * 
	 * @param featureName The name of the feature.
	 * @return The value of the feature.
	 */
	public StringRef getFeature(String featureName) {
		StringRef featureValue = features.get(featureName);
		if (featureValue == null) {
			featureValue = new StringRef();
			features.put(featureName, featureValue);
		}
		return featureValue;
	}
	
	/**
	 * Unifies this category with another category. Two categories can unify if and only if they have
	 * the same names and they have no features with conflicting values. If the unification fails, a
	 * UnificationFailedException is thrown. In this case, the two categories remain partly unified,
	 * i.e. no backtracking is done. Thus, this operation should be perfomed only if it is certain that
	 * the unification succeeds, or if the operation is performed on copies of objects that are not
	 * used anymore afterwards.
	 * 
	 * @param category The category to be unified with this category.
	 * @throws UnificationFailedException If unification fails.
	 */
	public void unify(Category category) throws UnificationFailedException {
		if (!name.equals(category.name)) {
			throw new UnificationFailedException();
		}
		for (String f : features.keySet()) {
			getFeature(f).unify(category.getFeature(f));
		}
		for (String f : category.features.keySet()) {
			getFeature(f).unify(category.getFeature(f));
		}
	}
	
	/**
	 * Tries to unify this category with another category. If unification is not possible, an exception
	 * is thrown. In the case unification would be possible, the unification is not performed completely.
	 * In any case the two categories remain in an unconsistent state afterwards. Thus, this operation should
	 * be performed only on copies of objects that are not used anymore afterwards.
	 * 
	 * @param category The category to be unified with this category.
	 * @throws UnificationFailedException If unification fails.
	 */
	public void tryToUnify(Category category) throws UnificationFailedException {
		if (!name.equals(category.name)) {
			throw new UnificationFailedException();
		}
		for (String f : features.keySet()) {
			features.get(f).unify(category.getFeature(f));
		}
	}
	
	/**
	 * Skolemizes the feature values of this category.
	 */
	public void skolemize() {
		for (String feature : features.keySet()) {
			StringRef s = features.get(feature);
			if (s.getString() == null) {
				try {
					s.unify(new StringRef("$SK" + skNumber++));
				} catch (UnificationFailedException ex) {}
			}
		}
	}
	
	/**
	 * Creates a deep copy of this category.
	 * 
	 * @return A deep copy.
	 */
	public Category deepCopy() {
		return deepCopy(new HashMap<Integer, StringEntity>());
	}
	
	/**
	 * Creates a deep copy of this category using the given string entities. This method is
	 * usually called form another deepCopy-method.
	 * 
	 * @param entities The string entities to be used.
	 * @return A deep copy.
	 */
	public Category deepCopy(HashMap<Integer, StringEntity> entities) {
		Category c;
		if (this instanceof Terminal) {
			c = new Terminal(name);
		} else {
			c = new Nonterminal(name);
		}
		for (String feature : features.keySet()) {
			StringRef s = features.get(feature);
			StringEntity se = entities.get(s.getID());
			if (se != null) {
				c.setFeature(feature, se.newStringRef());
			} else {
				StringRef sr = new StringRef(s.getString());
				c.setFeature(feature, sr);
				entities.put(s.getID(), sr.getStringEntity());
			}
		}
		return c;
	}
	
	public boolean equals(Object obj) {
		if (!(obj instanceof Category)) return false;
		return toString().equals(obj.toString());
	}
	
	public String toString() {
		String s = name;
		Set<String> featureKeys = features.keySet();
		if (featureKeys.size() > 0) s += "(";
		for (String feature : features.keySet()) {
			StringRef sr = features.get(feature);
			s += feature + ":";
			if (sr.getString() == null) {
				s += sr.getID();
			} else {
				s += sr.getString();
			}
			s += ",";
		}
		if (featureKeys.size() > 0) {
			s = s.substring(0, s.length()-1) + ")";
		}
		return s;
	}

}
