% This file is part of the Attempto Java Packages.
% Copyright 2008, Attempto Group, University of Zurich (see http://attempto.ifi.uzh.ch).
%
% The Attempto Java Packages is free software: you can redistribute it and/or modify it under the
% terms of the GNU Lesser General Public License as published by the Free Software Foundation,
% either version 3 of the License, or (at your option) any later version.
%
% The Attempto Java Packages is distributed in the hope that it will be useful, but WITHOUT ANY
% WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
% PURPOSE. See the GNU Lesser General Public License for more details.
%
% You should have received a copy of the GNU Lesser General Public License along with the Attempto
% Java Packages. If not, see http://www.gnu.org/licenses/.


:- module(generate_dcg, [
		generate_dcg/2  % +InputFile, +OutputFile
	]).

/** <module> Generate DCG Grammar

This module transfroms from the Attempto Chartparser Grammar Notation (ACGN) into Prolog DCG grammars.
Usage:

==
swipl -s generate_dcg.pl -g "generate_dcg('acgn.pl', 'dcg.pl')" -t halt
==

You may have to replace "swipl" by the name of your SWI Prolog executable.

@author Tobias Kuhn
@version 2008-05-25
*/


:- op(1200, xfx, '=>').
:- op(1200, xfx, '~>').
:- op(600, xfx, ':').
:- op(500, fx, '#').


%% generate_dcg(+InputFile, +OutputFile)
%
% This predicate reads the ACGN grammar from the input file and transforms it into a Prolog DCG grammar.

generate_dcg(InputFile, OutputFile) :-
    open(InputFile, read, In),
    load_terms(In),
    close(In),
    open(OutputFile, write, Out),
    write(Out, '% This code is automatically generated by the SWI Prolog program "generate_dcg.pl".\n\n'),
    process_terms(Out),
    close(Out).


:- dynamic(term/1).
:- dynamic(feature/1).


load_terms(In) :-
	read_term(In, Term, [module(generate_dcg)]),
	( Term == end_of_file ->
		true
	;
		load_term(Term),
		load_terms(In)
	).


load_term(Head => Body) :-
    load_cond(Head),
    load_conds(Body),
    !,
    assert(term(Head => Body)).

load_term(Head ~> Body) :-
    load_cond(Head),
    load_conds(Body),
    !,
    assert(term(Head ~> Body)).

load_term(Term) :-
    format(user_error, 'ERROR. Illegal term: ~q\n', Term).


load_conds((Cond,Rest)) :-
    !,
    load_cond(Cond),
    load_conds(Rest).

load_conds(Cond) :-
    load_cond(Cond).


load_cond(List) :-
    is_list(List),
    !.

load_cond(Cond) :-
    Cond =.. [_|Args],
    load_args(Args).


load_args([]).

load_args([Arg|Rest]) :-
    load_arg(Arg),
    load_args(Rest).


load_arg(Feature:Value) :-
    atomic(Feature),
    \+ compound(Value),
    !,
    load_feature(Feature).

load_arg(Term) :-
    format(user_error, 'ERROR. Illegal argument: ~q\n', Term),
    fail.


load_feature(Feature) :-
    feature(Feature),
    !.

load_feature(Feature) :-
    assert(feature(Feature)).


process_terms(Out) :-
    term(TermIn),
    transform_term(TermIn, TermOut),
    numbervars(TermOut, 0, _),
    format(Out, '~q.\n', TermOut),
    fail.

process_terms(_).


transform_term(Head => Body, HeadT --> BodyT) :-
    transform_cond(Head, HeadT),
    transform_conds(Body, BodyT).

transform_term(Head ~> Body, HeadT --> BodyT) :-
    transform_cond(Head, HeadT),
    transform_conds(Body, BodyT).


transform_conds((Cond,Rest), (CondT,RestT)) :-
    !,
    transform_cond(Cond, CondT),
    transform_conds(Rest, RestT).

transform_conds(Cond, CondT) :-
    transform_cond(Cond, CondT).


transform_cond(List, List) :-
    is_list(List),
    !.

transform_cond(Cond, CondT) :-
    Cond =.. [Pred|Args],
    get_features(Features),
    transform_args(Args, Features),
    transform_features(Features, FeaturesT),
    CondT =.. [Pred,FeaturesT].


transform_args([], _).

transform_args([Arg|Rest], Features) :-
    member(Arg, Features),
    !,
    transform_args(Rest, Features).

transform_args([Arg|_], _) :-
    format(user_error, 'ERROR. Illegal argument: ~q\n', Arg),
    fail.


get_features(Features) :-
    findall(F:_, feature(F), Features).


transform_features([], []).

transform_features([_:Value|Rest], [Value|RestT]) :-
    transform_features(Rest, RestT).
