// This file is part of the Attempto Java Packages.
// Copyright 2008, Attempto Group, University of Zurich (see http://attempto.ifi.uzh.ch).
//
// The Attempto Java Packages is free software: you can redistribute it and/or modify it under the
// terms of the GNU Lesser General Public License as published by the Free Software Foundation,
// either version 3 of the License, or (at your option) any later version.
//
// The Attempto Java Packages is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public License along with the Attempto
// Java Packages. If not, see http://www.gnu.org/licenses/.

package ch.uzh.ifi.attempto.owl;

import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.httpclient.NameValuePair;
import org.apache.commons.httpclient.methods.PostMethod;

/**
 * This is a simple interface to the OWL verbalizer.
 * OWL verbalizer is a tool that translates an OWL 2 ontology into an ACE text.
 * It expects the ontology to be represented in the OWL 2 XML serialization syntax.
 * The version of the OWL 2 XML syntax that the OWL verbalizer supports is described in
 * 
 * <blockquote>
 * <a href="http://www.w3.org/TR/2008/WD-owl2-xml-serialization-20080411/">http://www.w3.org/TR/2008/WD-owl2-xml-serialization-20080411/</a>
 * </blockquote>
 * 
 * The OWL verbalizer accepts one parameter, <code>xml</code>, the value of which
 * is the complete ontology as string.
 * The OWL verbalizer returns the ACE representation of the ontology as plain text (string).
 * In case the verbalizer fails to process an OWL axiom in the ontology,
 * an error message is returned between ACE comment symbols.
 * 
 * Note that future versions of the OWL verbalizer (and as a result this interface)
 * will expect a different input format (because the OWL 2 XML specification will be updated)
 * and probably produce a different output format (a more structured one, where e.g. error
 * messages are represented differently).
 * 
 * @author Kaarel Kaljurand
 *
 */
public class VerbalizerWebservice {

	private HttpClient client = new HttpClient();
	private String url = null;

	/**
	 * Constructs a new <code>VerbalizerWebservice</code> object
	 * on the basis of the URL of the OWL verbalizer webservice.
	 * 
	 * @param url The URL of the OWL verbalizer webservice.
	 */
	public VerbalizerWebservice(String url) {
		this.url = url;
	}

	/**
	 * Calls the OWL verbalizer webservice by giving the string representation of
	 * an XML-formatted ontology as input. Returns the corresponding ACE text
	 * with possible error messages as comments.
	 * 
	 * @param xml XML-representation of an OWL ontology.
	 * @return ACE text that corresponds to the ontology.
	 */
	public String call(String xml) {
		String acetextAsString = null;

		PostMethod method = new PostMethod(url);

		method.setRequestBody(new NameValuePair[] {
				new NameValuePair("xml", xml)
		});

		try {
			int statusCode = client.executeMethod(method);

			if (statusCode == HttpStatus.SC_OK) {
				acetextAsString = method.getResponseBodyAsString();
			}
			else {
				throw new RuntimeException("HTTP request failed: " + method.getStatusLine());
			}
		} catch (Exception e) {
			throw new RuntimeException("Error accessing OWL->ACE webservice: " + e.getMessage());
		} finally {
			method.releaseConnection();
		}

		return acetextAsString;
	}
}