// This file is part of the Attempto Java Packages.
// Copyright 2008, Attempto Group, University of Zurich (see http://attempto.ifi.uzh.ch).
//
// The Attempto Java Packages is free software: you can redistribute it and/or modify it under the
// terms of the GNU Lesser General Public License as published by the Free Software Foundation,
// either version 3 of the License, or (at your option) any later version.
//
// The Attempto Java Packages is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public License along with the Attempto
// Java Packages. If not, see http://www.gnu.org/licenses/.

package ch.uzh.ifi.attempto.preditor;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import nextapp.echo2.app.Alignment;
import nextapp.echo2.app.ApplicationInstance;
import nextapp.echo2.app.Border;
import nextapp.echo2.app.Button;
import nextapp.echo2.app.Color;
import nextapp.echo2.app.Column;
import nextapp.echo2.app.Extent;
import nextapp.echo2.app.Font;
import nextapp.echo2.app.Insets;
import nextapp.echo2.app.Row;
import nextapp.echo2.app.SplitPane;
import nextapp.echo2.app.TextArea;
import nextapp.echo2.app.event.ActionEvent;
import nextapp.echo2.app.event.ActionListener;
import ch.uzh.ifi.attempto.chartparser.ChartParser;
import ch.uzh.ifi.attempto.chartparser.Grammar;
import ch.uzh.ifi.attempto.echocomp.GeneralButton;
import ch.uzh.ifi.attempto.echocomp.Label;
import ch.uzh.ifi.attempto.echocomp.Style;
import ch.uzh.ifi.attempto.echocomp.TextField;
import ch.uzh.ifi.attempto.echocomp.WindowPane;
import ch.uzh.ifi.attempto.preditor.text.TextContainer;
import ch.uzh.ifi.attempto.preditor.text.TextElement;
import echopointng.KeyStrokeListener;

/**
 * This class represents a predictive editor window. The predictive editor enables easy creation of texts that
 * comply with a certain grammar. The users can create such a text word-by-word by clicking on one of different
 * menu items. The menu items are structured into menu blocks each of which has a name that is displayed above
 * the menu block.
 * 
 * @author Tobias Kuhn
 */
public class PreditorWindow extends WindowPane implements ActionListener {
	
	private static final long serialVersionUID = -7815494421993305554L;
	
	private List<MenuBlockContent> menuBlockContents = new ArrayList<MenuBlockContent>();
	private List<MenuBlock> menuBlocksTop = new ArrayList<MenuBlock>();
	private List<MenuBlock> menuBlocksBottom = new ArrayList<MenuBlock>();
	private List<SplitPane> menuSplitPanesTop = new ArrayList<SplitPane>();
	private List<SplitPane> menuSplitPanesBottom = new ArrayList<SplitPane>();
	
	private TextArea textArea = new TextArea();
	private TextField textField;
	private SplitPane menuBlockPane = new SplitPane(SplitPane.ORIENTATION_HORIZONTAL_LEFT_RIGHT, new Extent(0));
	private SplitPane doubleColumnMenuPane =  new SplitPane(SplitPane.ORIENTATION_VERTICAL_TOP_BOTTOM, new Extent(258));
	private GeneralButton deleteButton = new GeneralButton("< Delete", 70, this);
	
	private KeyStrokeListener keyStrokeListener = new KeyStrokeListener();
	
	private TextContainer newTextContainer = new TextContainer();
	private MenuCreator menuCreator;
	
	private ChartParser parser;
	
	private Button okButton = new GeneralButton("OK", 70, this);
	private Button cancelButton = new GeneralButton("Cancel", 70, this);
	
	private ArrayList<ActionListener> actionListeners = new ArrayList<ActionListener>();
	
	/**
	 * Creates a new predictive editor window for the given grammar using the given menu creator.
	 * 
	 * @param title The title of the window.
	 * @param grammar The grammar to be used.
	 * @param menuCreator The menu creator to be used.
	 */
	public PreditorWindow(String title, Grammar grammar, MenuCreator menuCreator) {
		this.parser = new ChartParser(grammar);
		this.menuCreator = menuCreator;
		
		setModal(true);
		setTitle(title);
		setTitleFont(new Font(Style.fontTypeface, Font.ITALIC, new Extent(13)));
		setWidth(new Extent(753));
		setHeight(new Extent(503));
		setResizable(false);
		setTitleBackground(Style.windowTitleBackground);
		setStyleName("Default");
		
		Row buttonBar = new Row();
		buttonBar.setAlignment(new Alignment(Alignment.RIGHT, Alignment.CENTER));
		buttonBar.setInsets(new Insets(10, 17, 10, 10));
		buttonBar.setCellSpacing(new Extent(5));
		buttonBar.add(okButton);
		buttonBar.add(cancelButton);
		
		SplitPane splitPane = new SplitPane(SplitPane.ORIENTATION_VERTICAL_BOTTOM_TOP, new Extent(47));
		splitPane.add(buttonBar);
		add(splitPane);
		
		SplitPane editorPane = new SplitPane(SplitPane.ORIENTATION_VERTICAL_TOP_BOTTOM, new Extent(148));
		
		Column textColumn = new Column();
		textColumn.setInsets(new Insets(10, 10, 0, 0));
		textColumn.setCellSpacing(new Extent(10));
		
		Column textAreaColumn = new Column();
		
		textArea.setWidth(new Extent(702));
		textArea.setHeight(new Extent(42));
		textArea.setEnabled(false);
		textArea.setFont(new Font(Style.fontTypeface, Font.PLAIN, new Extent(12)));
		textArea.setBorder(new Border(1, new Color(180, 180, 180), Border.STYLE_SOLID));
		textArea.setInsets(new Insets(4, 4));
		textArea.setBackground(new Color(255, 255, 255));
		textArea.setFocusTraversalParticipant(false);
		textAreaColumn.add(textArea);
		
		Row textAreaButtonBar = new Row();
		textAreaButtonBar.setAlignment(new Alignment(Alignment.RIGHT, Alignment.CENTER));
		textAreaButtonBar.setInsets(new Insets(0, 5, 10, 0));
		textAreaButtonBar.setCellSpacing(new Extent(5));
		deleteButton.setFocusTraversalParticipant(false);
		textAreaButtonBar.add(deleteButton);
		textAreaColumn.add(textAreaButtonBar);
		
		textColumn.add(textAreaColumn);
		
		Column textFieldColumn = new Column();
		textFieldColumn.setCellSpacing(new Extent(1));
		Label textFieldLabel = new Label("text", Font.ITALIC, 11);
		textFieldColumn.add(textFieldLabel);
		
		textField = new TextField(this);
		textField.setWidth(new Extent(708));
		textField.setFocusTraversalParticipant(true);
		textField.setFocusTraversalIndex(0);
		textField.setDisabledBackground(Style.lightDisabled);
		Row textFieldRow = new Row();
		textFieldRow.add(textField);
		TextField dummyTextField = new TextField();
		dummyTextField.setWidth(new Extent(1));
		dummyTextField.setBorder(new Border(0, null, 0));
		dummyTextField.setBackground(Color.WHITE);
		textFieldRow.add(dummyTextField);
		textFieldColumn.add(textFieldRow);
		
		keyStrokeListener.addKeyCombination(KeyStrokeListener.VK_TAB, "Tab");
		keyStrokeListener.addKeyCombination(KeyStrokeListener.VK_RIGHT, "Tab");
		keyStrokeListener.addActionListener(this);
		textFieldColumn.add(keyStrokeListener);
		
		textColumn.add(textFieldColumn);
		editorPane.add(textColumn);
		
		menuBlockPane.setSeparatorWidth(new Extent(10));
		menuBlockPane.setSeparatorColor(Color.WHITE);
		menuBlockPane.add(new Label());
		
		SplitPane parentSplitPane = doubleColumnMenuPane;
		for (int i=0; i<10; i++) {
			MenuBlock menuBlock = new MenuBlock(this, this);
			menuBlocksTop.add(menuBlock);
			SplitPane menuSplitPane = new SplitPane(SplitPane.ORIENTATION_HORIZONTAL_LEFT_RIGHT);
			menuSplitPane.setSeparatorWidth(new Extent(10));
			menuSplitPane.setSeparatorColor(Color.WHITE);
			menuSplitPane.setVisible(false);
			menuSplitPane.add(menuBlock);
			menuSplitPanesTop.add(menuSplitPane);
			parentSplitPane.add(menuSplitPane);
			parentSplitPane = menuSplitPane;
		}
		
		parentSplitPane = doubleColumnMenuPane;
		for (int i=0; i<10; i++) {
			MenuBlock menuBlock = new MenuBlock(this, this);
			menuBlocksBottom.add(menuBlock);
			SplitPane menuSplitPane = new SplitPane(SplitPane.ORIENTATION_HORIZONTAL_LEFT_RIGHT);
			menuSplitPane.setSeparatorWidth(new Extent(10));
			menuSplitPane.setSeparatorColor(Color.WHITE);
			menuSplitPane.setVisible(false);
			menuSplitPane.add(menuBlock);
			menuSplitPanesBottom.add(menuSplitPane);
			parentSplitPane.add(menuSplitPane);
			parentSplitPane = menuSplitPane;
		}
		
		doubleColumnMenuPane.setSeparatorHeight(new Extent(12));
		doubleColumnMenuPane.setSeparatorColor(Color.WHITE);
		menuBlockPane.add(doubleColumnMenuPane);
		editorPane.add(menuBlockPane);
		splitPane.add(editorPane);
		
		update();
	}
	
	/**
	 * Returns the (partial) text that has been entered .
	 * 
	 * @return The (partial) text in the form of a text container.
	 */
	public TextContainer getTextContainer() {
		return newTextContainer;
	}
	
	/**
	 * Returns a text element with the given content if it is a possible next token.
	 * 
	 * @param content The content of the text element.
	 * @return The text element.
	 */
	public TextElement getPossibleNextToken(String content) {
		for (MenuBlockContent m : menuBlockContents) {
			TextElement e = m.getEntry(content);
			if (e != null) return e;
		}
		return null;
	}
	
	/**
	 * Adds the text element to the end of the text.
	 * 
	 * @param te The text element to be added.
	 */
	public void addTextElement(TextElement te) {
		textElementSelected(te);
		textField.setText("");
		update();
	}
	
	/**
	 * Reads the text and adds it to the end of the current text as far as possible.
	 * 
	 * @param text The text to be added.
	 */
	public void addText(String text) {
		handleTextInput(tokenize(text));
		update();
	}
	
	private void textElementSelected(TextElement te) {
		newTextContainer.addElement(te);
		parser.addToken(te.getCategory());
		log("edit", "words added: " + te.getText());
	}
	
	private int getElementsCount() {
		int c = 0;
		for (MenuBlockContent m : menuBlockContents) {
			c += m.getEntries().size();
		}
		return c;	
	}
	
	private String getStartString() {
		String startString = "";
		ArrayList<String> blockStartStrings = new ArrayList<String>();
		
		for (MenuBlockContent mc : menuBlockContents) {
			String s = mc.getStartString();
			if (s != null) {
				blockStartStrings.add(s);
			}
		}
		
		if (blockStartStrings.isEmpty()) return null;
		
		String first = blockStartStrings.get(0);
		blockStartStrings.remove(0);
		
		if (blockStartStrings.isEmpty()) return first;
		
		for (int i = 0; i < first.length(); i++) {
			char c = first.charAt(i);
			boolean stop = false;
			for (String s : blockStartStrings) {
				if (s.length() <= i || s.charAt(i) != c) stop = true;
			}
			if (stop) break;
			startString += c;
		}
		
		return startString;
	}
	
	private void update() {
		updateMenuBlockContents();
		setFilter(textField.getText());
		int mbCount = menuBlockContents.size();
		if (mbCount < 5) {
			int width = ( 720 / ( mbCount > 3 ? mbCount : 3 ) ) - 10;
			for (int i=0; i < menuBlocksTop.size(); i++) {
				if (menuBlockContents.size() > i) {
					menuBlocksTop.get(i).setContent(menuBlockContents.get(i), width, 16);
					menuSplitPanesTop.get(i).setSeparatorPosition(new Extent(width));
					menuSplitPanesTop.get(i).setVisible(true);
				} else {
					menuSplitPanesTop.get(i).setVisible(false);
				}
			}
			doubleColumnMenuPane.setSeparatorPosition(new Extent(258));
		} else {
			int firstRowCount = (mbCount + 1) / 2;
			int width = ( 720 / firstRowCount ) - 10;
			for (int i=0; i < menuBlocksTop.size(); i++) {
				if (i < firstRowCount) {
					menuBlocksTop.get(i).setContent(menuBlockContents.get(i), width, 7);
					menuSplitPanesTop.get(i).setSeparatorPosition(new Extent(width));
					menuSplitPanesTop.get(i).setVisible(true);
				} else {
					menuSplitPanesTop.get(i).setVisible(false);
				}
			}
			for (int i=0; i < menuBlocksBottom.size(); i++) {
				if (firstRowCount + i < mbCount) {
					menuBlocksBottom.get(i).setContent(menuBlockContents.get(firstRowCount + i), width, 7);
					menuSplitPanesBottom.get(i).setSeparatorPosition(new Extent(width));
					menuSplitPanesBottom.get(i).setVisible(true);
				} else {
					menuSplitPanesBottom.get(i).setVisible(false);
				}
			}
			doubleColumnMenuPane.setSeparatorPosition(new Extent(123));
		}
		textField.setEnabled(menuBlockContents.size() > 0 || !textField.getText().equals(""));
		ApplicationInstance.getActive().setFocusedComponent(textField);
		deleteButton.setEnabled(textArea.getText().length() > 0);
	}
	
	private void updateMenuBlockContents() {
		textArea.setText(newTextContainer.getText());
		menuBlockContents.clear();
		List<MenuBlockContent> newMenuBlockContents = menuCreator.createMenu(parser, newTextContainer);
		for (MenuBlockContent c : newMenuBlockContents) {
			if (!c.isEmpty()) {
				menuBlockContents.add(c);
			}
		}
	}
	
	private void setFilter(String filter) {
		for (MenuBlockContent c : menuBlockContents) {
			c.setFilter(filter);
		}
	}
	
	private void handleTextInput() {
		handleTextInput(tokenize(textField.getText()));
	}
	
	private void handleTextInput(ArrayList<String> tokens) {
		
		TextElement t = null;
		String s = "";
		int c = 0;
		
		for (String token : tokens) {
			setFilter(s + token);
			TextElement t2 = getPossibleNextToken(s + token);
			int elCount = getElementsCount();
			if (t2 != null) {
				if (elCount == 1 && c < tokens.size()-1) {
					t = t2;
					c++;
					break;
				} else if (c < tokens.size()-1) {
					t = t2;
				}
			}
			if (elCount == 0) {
				break;
			} else if (c == tokens.size()-1) { // && token.length() > 0) {
				t = null;
				c++;
				break;
			} else {
				s += token + " ";
				c++;
			}
		}
		
		setFilter(null);
		
		if (t != null) {
			textElementSelected(t);
			updateMenuBlockContents();
			for (int i=0; i<c; i++) tokens.remove(0);
			handleTextInput(tokens);
		} else {
			String text = "";
			for (String token : tokens) {
				if (text.length() > 0) text += " ";
				text += token;
			}
			textField.setText(text);
		}
	}
	
	private ArrayList<String> tokenize(String text) {
		text = text.replaceAll("\\.", " . ");
		text = text.replaceAll("\\?", " ? ");
		text = text.replaceAll("\\!", " ! ");
		
		ArrayList<String> tokens = new ArrayList<String>(Arrays.asList(text.split(" ")));
		
		while (tokens.contains("")) {
			tokens.remove("");
		}
		
		if (text.endsWith(" ")) {
			tokens.add("");
		}
		
		return tokens;
	}
	
	/**
	 * Adds a new action-listener.
	 * 
	 * @param actionListener The new action-listener.
	 */
	public void addActionListener(ActionListener actionListener) {
		actionListeners.add(actionListener);
	}
	
	/**
	 * Removes the action-listener.
	 * 
	 * @param actionListener The action-listener to be removed.
	 */
	public void removeActionListener(ActionListener actionListener) {
		actionListeners.remove(actionListener);
	}
	
	/**
	 * Removes all action-listeners.
	 */
	public void removeAllActionListeners() {
		actionListeners.clear();
	}
	
	private void notifyActionListeners(ActionEvent event) {
		for (ActionListener al : actionListeners) {
			al.actionPerformed(event);
		}
	}
	
	public void actionPerformed(ActionEvent e) {
		if (e.getSource() == cancelButton) {
			log("edit", "pressed: cancel");
			notifyActionListeners(new ActionEvent(this, "Cancel"));
			return;
		} else if (e.getSource() == okButton) {
			log("edit", "pressed: ok");
			handleTextInput();
			update();
			notifyActionListeners(new ActionEvent(this, "OK"));
			return;
		} else if (e.getSource() == deleteButton) {
			log("edit", "pressed: < delete");
			newTextContainer.removeLastElement();
			parser.removeToken();
			textField.setText("");
		} else if (e.getSource() instanceof MenuEntry) {
			TextElement te = ((MenuEntry) e.getSource()).getTextElement();
			log("edit", "pressed: menu-entry " + te.getText());
			textElementSelected(te);
			textField.setText("");
		} else if (e.getSource() == textField) {
			log("edit", "pressed: enter-key");
			handleTextInput();
			TextElement te = getPossibleNextToken(textField.getText());
			if (te != null) {
				textElementSelected(te);
				textField.setText("");
			}
		} else if ("Tab".equals(e.getActionCommand())) {
			log("edit", "pressed: tab-key");
			handleTextInput();
		}
		
		update();
		
		if ("Tab".equals(e.getActionCommand())) {
			String s = getStartString();
			if (s != null) textField.setText(s);
		}
	}
	
	private void log(String type, String text) {
		// No logging done at the moment
	}
	
	public String toString() {
		return "sentence: " + newTextContainer.getText();
	}
	
}
