// This file is part of the Attempto Java Packages.
// Copyright 2008, Attempto Group, University of Zurich (see http://attempto.ifi.uzh.ch).
//
// The Attempto Java Packages is free software: you can redistribute it and/or modify it under the
// terms of the GNU Lesser General Public License as published by the Free Software Foundation,
// either version 3 of the License, or (at your option) any later version.
//
// The Attempto Java Packages is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public License along with the Attempto
// Java Packages. If not, see http://www.gnu.org/licenses/.

package ch.uzh.ifi.attempto.acewiki.gui;

import java.util.Collections;
import java.util.List;

import nextapp.echo2.app.Color;
import nextapp.echo2.app.Column;
import nextapp.echo2.app.Component;
import nextapp.echo2.app.Font;
import nextapp.echo2.app.Insets;
import nextapp.echo2.app.ResourceImageReference;
import nextapp.echo2.app.Row;
import nextapp.echo2.app.event.ActionEvent;
import nextapp.echo2.app.event.ActionListener;
import ch.uzh.ifi.attempto.acewiki.Task;
import ch.uzh.ifi.attempto.acewiki.Wiki;
import ch.uzh.ifi.attempto.acewiki.core.ontology.Individual;
import ch.uzh.ifi.attempto.acewiki.core.ontology.OntologyElement;
import ch.uzh.ifi.attempto.acewiki.core.ontology.Sentence;
import ch.uzh.ifi.attempto.acewiki.core.text.OntologyTextElement;
import ch.uzh.ifi.attempto.acewiki.gui.editor.SentenceEditorHandler;
import ch.uzh.ifi.attempto.acewiki.gui.page.ArticlePage;
import ch.uzh.ifi.attempto.acewiki.gui.page.LogicPage;
import ch.uzh.ifi.attempto.acewiki.gui.page.SentencePage;
import ch.uzh.ifi.attempto.acewiki.gui.page.WikiPage;
import ch.uzh.ifi.attempto.echocomp.DelayedComponent;
import ch.uzh.ifi.attempto.echocomp.HSpace;
import ch.uzh.ifi.attempto.echocomp.Label;
import ch.uzh.ifi.attempto.echocomp.MessageWindow;
import ch.uzh.ifi.attempto.echocomp.SolidLabel;
import ch.uzh.ifi.attempto.echocomp.VSpace;
import ch.uzh.ifi.attempto.preditor.text.TextElement;

/**
 * This class represents a text row that consists of a drop down menu and an ACE text.
 * 
 * @author Tobias Kuhn
 */
public class TextRow extends Column implements ActionListener {

	private static final long serialVersionUID = -540135972060005725L;
	
	private Sentence sentence;
	private Wiki wiki;
	private WikiPage hostPage;
	
	private Row sentenceRow = new Row();
	private DropDownMenu dropDown;
	
	/**
	 * Creates a new text row. The host page is the page that contains the text row (which is
	 * not necessarily the owner page of the sentence).
	 * 
	 * @param sentence The sentence to be shown.
	 * @param hostPage The host page of the text row.
	 */
	public TextRow(Sentence sentence, WikiPage hostPage) {
		this.sentence = sentence;
		this.hostPage = hostPage;
		this.wiki = hostPage.getWiki();
		update();
	}
	
	private void update() {
		if (sentence.isInferred()) {
			dropDown = new DropDownMenu("light-blue", this);
		} else if (sentence.isReasonerParticipant() || (sentence.isQuestion() && sentence.isOWL()) ) {
			dropDown = new DropDownMenu("blue", this);
		} else {
			dropDown = new DropDownMenu("red", this);
		}
		if (!sentence.isIntegrated() && !sentence.isInferred()) {
			dropDown.addMenuEntry("Reassert");
			dropDown.addMenuSeparator();
		}
		if (!sentence.isInferred()) {
			dropDown.addMenuEntry("Edit...");
		}
		if (hostPage instanceof ArticlePage) {
			dropDown.addMenuEntry("Add...");
		}
		if (!sentence.isInferred()) {
			dropDown.addMenuEntry("Delete");
		}
		dropDown.addMenuSeparator();
		dropDown.addMenuEntry("Details");
		dropDown.addMenuEntry("Logic");
		
		Row r = new Row();
		Color color = Color.BLACK;
		boolean isRed = !sentence.isIntegrated() && !sentence.isInferred() && !sentence.isQuestion();
		if (isRed) {
			color = new Color(193, 0, 0);
		}
		for (TextElement e : sentence.getTextElements()) {
			if (!e.getText().matches("[.?]") && r.getComponentCount() > 0) {
				r.add(new HSpace());
			}
			if (e instanceof OntologyTextElement) {
				// Proper names with definite articles are handled differently:
				// The "the" is not a part of the link. Probably, this should be done at a different place...
				OntologyTextElement ote = (OntologyTextElement) e;
				OntologyElement oe = ote.getOntologyElement();
				if (oe instanceof Individual) {
					Individual ind = (Individual) oe;
					int wn = ote.getWordNumber();
					
					if (ind.hasDefiniteArticle(wn)) {
						SolidLabel l = new SolidLabel(e.getText().substring(0, 3));
						l.setForeground(color);
						r.add(l);
						r.add(new HSpace());
						r.add(new WikiLink(oe, oe.getPrettyWord(wn + 1), wiki, isRed));
					} else {
						r.add(new WikiLink(((OntologyTextElement) e), wiki, isRed));
					}
				} else {
					r.add(new WikiLink(((OntologyTextElement) e), wiki, isRed));
				}
			} else {
				SolidLabel l = new SolidLabel(e.getText());
				l.setForeground(color);
				r.add(l);
			}
		}
		
		removeAll();
		sentenceRow.removeAll();
		sentenceRow.add(dropDown);
		sentenceRow.add(new HSpace(5));
		sentenceRow.add(r);
		sentenceRow.add(new HSpace(10));
		add(sentenceRow);
		
		if (sentence.isQuestion() && hostPage instanceof ArticlePage) {
			
			Column answerColumn = new Column();
			answerColumn.setInsets(new Insets(20, 0, 0, 0));
			add(answerColumn);
			
			if (sentence.isAnswerCached()) {
				
				Column column = new Column();
				List<Individual> individuals = sentence.getAnswer();
				if (individuals == null) {
					column.add(new SolidLabel("(invalid question)", Font.ITALIC, 10));
				} else if (individuals.size() > 0) {
					Collections.sort(individuals);
					for (Individual ind : individuals) {
						Row answerRow = new Row();
						answerRow.add(new ListItem(new WikiLink(ind, wiki)));
						column.add(answerRow);
					}
				} else {
					column.add(new SolidLabel("(no answer found)", Font.ITALIC, 10));
				}
				column.add(new VSpace(4));
				answerColumn.add(column);
				
			} else {
				answerColumn.add(new DelayedComponent(new Label(new ResourceImageReference("ch/uzh/ifi/attempto/acewiki/gui/img/wait.gif"))) {
					
					private static final long serialVersionUID = 7865984138467729544L;

					public Component initComponent() {
						Column column = new Column();
						List<Individual> individuals = sentence.getAnswer();
						if (individuals == null) {
							column.add(new SolidLabel("(invalid question)", Font.ITALIC, 10));
						} else if (individuals.size() > 0) {
							Collections.sort(individuals);
							for (Individual ind : individuals) {
								Row answerRow = new Row();
								answerRow.add(new ListItem(new WikiLink(ind, wiki)));
								column.add(answerRow);
							}
						} else {
							column.add(new SolidLabel("(no answer found)", Font.ITALIC, 10));
						}
						column.add(new VSpace(4));
						
						return column;
					}
					
				});
			}
			
		}
	}

	public void actionPerformed(ActionEvent e) {
		if (e.getActionCommand().equals("Edit...")) {
			wiki.log("page", "dropdown: edit sentence: " + sentence.getText());
			ArticlePage page = ArticlePage.create(sentence.getOwner(), wiki);
			wiki.showPage(page);
			page.edit(sentence);
		} else if (e.getActionCommand().equals("Add...")) {
			wiki.log("page", "dropdown: add sentence");
			wiki.showWindow(SentenceEditorHandler.generatePreditorAddWindow(sentence, (ArticlePage) hostPage));
		} else if (e.getActionCommand().equals("Delete")) {
			wiki.log("page", "dropdown: delete sentence: " + sentence.getText());
			wiki.showWindow(new MessageWindow("Delete", "Do you really want to delete this sentence?", null, this, "Yes", "No"));
		} else if (e.getActionCommand().equals("Reassert")) {
			int success = sentence.reassert();
			if (success == 1) {
				wiki.showWindow(new MessageWindow("Conflict", "A sentence is in conflict with the current knowledge. For that reason, it cannot be added to the knowledge base.", "OK"));
			} else if (success == 2) {
				wiki.showWindow(new MessageWindow("Error", "A sentence could not be added to the knowledge base because the knowledge base got too complex.", "OK"));
			}
			if (sentence.isIntegrated()) {
				update();
			}
		} else if (e.getActionCommand().equals("Details")) {
			wiki.log("page", "dropdown: details sentence: " + sentence.getText());
			wiki.showPage(new SentencePage(wiki, sentence));
		} else if (e.getActionCommand().equals("Logic")) {
			wiki.log("page", "dropdown: logic sentence: " + sentence.getText());
			wiki.showPage(new LogicPage(wiki, sentence));
		} else if (e.getSource() instanceof MessageWindow && e.getActionCommand().equals("Yes")) {
			wiki.log("page", "dropdown: delete confirmed: " + sentence.getText());
			
			wiki.enqueueTaskShowingWaitWindow(
				"Updating",
				"The sentence is being removed from the knowledge base...",
				new Task() {
					public void run() {
						sentence.getOwner().remove(sentence);
					}
					public void updateGUI() {
						wiki.update();
						wiki.refresh();
					}
				}
			);
		}
	}

}
