// This file is part of the Attempto Java Packages.
// Copyright 2008, Attempto Group, University of Zurich (see http://attempto.ifi.uzh.ch).
//
// The Attempto Java Packages is free software: you can redistribute it and/or modify it under the
// terms of the GNU Lesser General Public License as published by the Free Software Foundation,
// either version 3 of the License, or (at your option) any later version.
//
// The Attempto Java Packages is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public License along with the Attempto
// Java Packages. If not, see http://www.gnu.org/licenses/.

package ch.uzh.ifi.attempto.acewiki.gui.editor;

import nextapp.echo2.app.event.ActionListener;
import ch.uzh.ifi.attempto.acewiki.Wiki;
import ch.uzh.ifi.attempto.acewiki.core.ontology.Individual;
import ch.uzh.ifi.attempto.acewiki.core.ontology.OntologyElement;
import ch.uzh.ifi.attempto.acewiki.core.text.TextElemFactory;
import ch.uzh.ifi.attempto.chartparser.Terminal;
import ch.uzh.ifi.attempto.preditor.MenuCreator;
import ch.uzh.ifi.attempto.preditor.MenuEntry;
import ch.uzh.ifi.attempto.preditor.SpecialMenuItem;
import ch.uzh.ifi.attempto.preditor.text.BasicTextElement;

/**
 * This is the menu creator class that generates the menu entries for the predictive editor
 * on the basis of the AceWiki grammar.
 * 
 * @author Tobias Kuhn
 */
class AceWikiMenuCreator extends MenuCreator {
	
	private Wiki wiki;
	private OntologyElement highlightedElement;
	private ActionListener actionListener;
	
	
	public AceWikiMenuCreator(Wiki wiki, OntologyElement highlightedElement, ActionListener actionListener) {
		this.wiki = wiki;
		this.highlightedElement = highlightedElement;
		this.actionListener = actionListener;
	}
	
	public void initMenuCreation() {
		prepareMenuBlock("function word", true);
		prepareMenuBlock("noun", true);
		prepareMenuBlock("plural noun", true);
		prepareMenuBlock("proper name", true);
		prepareMenuBlock("of-construct", true);
		prepareMenuBlock("verb", true);
		prepareMenuBlock("passive verb", true);
		prepareMenuBlock("transitive adjective", true);
		prepareMenuBlock("reference", false);
		prepareMenuBlock("new variable", false);
	}

	public void processCategory(Terminal category) {
		if (category.getName().equals("PROPERNAME")) {
			SpecialMenuItem newItem = new SpecialMenuItem("new...") {
				
				private static final long serialVersionUID = -350111458598367634L;

				public void action() {
					wiki.showWindow(ProperNameForm.createCreatorWindow(wiki, actionListener));
				}
				
			};
			addMenuItem("proper name", newItem);
			addMenuEntries("proper name", "Proper Name", 0);
			// abbreviations have to be added separately:
			for (OntologyElement el : wiki.getOntologyElements()) {
				if (!(el instanceof Individual)) continue;
				Individual ind = (Individual) el;
				if (ind.getAbbreviation() == null) continue;
				MenuEntry menuEntry = new MenuEntry(TextElemFactory.createTextElement(el, 2));
				menuEntry.setHighlighted(el == highlightedElement);
				addMenuItem("proper name", menuEntry);
			}
		} else if (category.getName().equals("NOUN")) {
			SpecialMenuItem newItem = new SpecialMenuItem("new...") {
				
				private static final long serialVersionUID = -1231084311228714943L;

				public void action() {
					wiki.showWindow(NounForm.createCreatorWindow(0, wiki, actionListener));
				}
				
			};
			addMenuItem("noun", newItem);
			addMenuEntries("noun", "Noun", 0);
		} else if (category.getName().equals("NOUNPL")) {
			SpecialMenuItem newItem = new SpecialMenuItem("new...") {
				
				private static final long serialVersionUID = -2417333739773492518L;

				public void action() {
					wiki.showWindow(NounForm.createCreatorWindow(1, wiki, actionListener));
				}
				
			};
			addMenuItem("plural noun", newItem);
			addMenuEntries("plural noun", "Noun", 1);
		} else if (category.getName().equals("NOUNOF")) {
			SpecialMenuItem newItem = new SpecialMenuItem("new...") {
				
				private static final long serialVersionUID = 1177809888334571322L;

				public void action() {
					wiki.showWindow(NounOfForm.createCreatorWindow(wiki, actionListener));
				}
				
			};
			addMenuItem("of-construct", newItem);
			addMenuEntries("of-construct", "Of-Construct", 0);
		} else if (category.getName().equals("VERB")) {
			SpecialMenuItem newItem = new SpecialMenuItem("new...") {
				
				private static final long serialVersionUID = -7240634464592044263L;

				public void action() {
					wiki.showWindow(VerbForm.createCreatorWindow(0, wiki, actionListener));
				}
				
			};
			addMenuItem("verb", newItem);
			addMenuEntries("verb", "Verb", 0);
		} else if (category.getName().equals("VERBINF")) {
			SpecialMenuItem newItem = new SpecialMenuItem("new...") {
				
				private static final long serialVersionUID = -3700453740738053228L;

				public void action() {
					wiki.showWindow(VerbForm.createCreatorWindow(1, wiki, actionListener));
				}
				
			};
			addMenuItem("verb", newItem);
			addMenuEntries("verb", "Verb", 1);
		} else if (category.getName().equals("PVERB")) {
			SpecialMenuItem newItem = new SpecialMenuItem("new...") {
				
				private static final long serialVersionUID = -1474626348181083481L;

				public void action() {
					wiki.showWindow(VerbForm.createCreatorWindow(2, wiki, actionListener));
				}
				
			};
			addMenuItem("passive verb", newItem);
			addMenuEntries("passive verb", "Verb", 2);
		} else if (category.getName().equals("TRADJ")) {
			SpecialMenuItem newItem = new SpecialMenuItem("new...") {
				
				private static final long serialVersionUID = 1928452310225044129L;

				public void action() {
					wiki.showWindow(TrAdjForm.createCreatorWindow(wiki, actionListener));
				}
				
			};
			addMenuItem("transitive adjective", newItem);
			addMenuEntries("transitive adjective", "Transitive Adjective", 0);
		} else if (category.getName().equals("REFERENCE")) {
			addReferenceEntries("reference", "REFERENCE");
		} else if (category.getName().equals("VARIABLE")) {
			addVariableEntries("new variable", 3, "VARIABLE");
		} else if (category.getName().equals("NUMBER")) {
			for (int i = 2; i <= 100; i++) {
				addMenuEntry("function word", new BasicTextElement(i, "NUMBER"));
			}
		} else {
			addMenuEntry("function word", new BasicTextElement(category.getName()));
		}
	}
	
	private void addMenuEntries(String menuBlockName, String type, int wordNumber) {
		for (OntologyElement el : wiki.getOntologyElements()) {
			if (!type.equals(el.getType())) continue;
			if (el.getWord(wordNumber) == null) continue;
			MenuEntry menuEntry = new MenuEntry(TextElemFactory.createTextElement(el, wordNumber));
			menuEntry.setHighlighted(el == highlightedElement);
			addMenuItem(menuBlockName, menuEntry);
		}
	}

}
