// This file is part of the Attempto Java Packages.
// Copyright 2008, Attempto Group, University of Zurich (see http://attempto.ifi.uzh.ch).
//
// The Attempto Java Packages is free software: you can redistribute it and/or modify it under the
// terms of the GNU Lesser General Public License as published by the Free Software Foundation,
// either version 3 of the License, or (at your option) any later version.
//
// The Attempto Java Packages is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public License along with the Attempto
// Java Packages. If not, see http://www.gnu.org/licenses/.

package ch.uzh.ifi.attempto.chartparser;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

/**
 * This is a chart parser (concretely an Earley parser) fully implemented in Java. However, there is a
 * Prolog format ("Attempto Chartparser Grammar Notation" or "ACGN") that can be transformed into Java
 * (at compile time).
 * 
 * @author Tobias Kuhn
 * @see Grammar
 */
public class ChartParser {
	
	private final Grammar grammar;
	private final Chart chart = new Chart();
	private final ArrayList<Terminal> tokens = new ArrayList<Terminal>();
	private boolean debug;
	
	/**
	 * Creates a new chart parser for the given grammar. The grammar must not be changed afterwards.
	 * 
	 * @param grammar The grammar to be used by the chart parser.
	 */
	public ChartParser(Grammar grammar) {
		this.grammar = grammar;
		init(grammar.getStartCategory());
		completeAndPredict();
	}
	
	/**
	 * This method can be used to switch on/off debug mode (default is off). In debug mode, messages about the actions
	 * of the chart parser are printed onto the standard error device.
	 * 
	 * @param debug true to switch on debug mode or false to switch it off.
	 */
	public void debug(boolean debug) {
		this.debug = debug;
	}
	
	/**
	 * Adds the token to the token sequence and makes one more parsing step to process it.
	 * 
	 * @param token The token to be added to the token sequence.
	 */
	public void addToken(String token) {
		Terminal t = new Terminal(token);
		tokens.add(t);
		Edge edge = new Edge(tokens.size()-1, tokens.size(), t, true);
		chart.addEdge(edge);
		if (debug) System.err.println("SCANNER: " + edge);
		completeAndPredict();
		//if (debug) System.err.println("CHART:");
		//if (debug) System.err.println(chart);
	}
	
	/**
	 * Adds the tokens to the token sequence and processes them.
	 * 
	 * @param tokens The tokens to be added to the token sequence.
	 */
	public void addTokens(List<String> tokens) {
		for (String s : tokens) {
			addToken(s);
		}
	}
	
	/**
	 * Removes the last token and reverts the last parsing step.
	 */
	public void removeToken() {
		chart.removeEdgesWithEndPos(tokens.size());
		tokens.remove(tokens.size()-1);
	}
	
	/**
	 * Removes all tokens in the current token sequence and resets the chart.
	 */
	public void removeAllTokens() {
		tokens.clear();
		chart.clear();
		init(grammar.getStartCategory());
		completeAndPredict();
	}
	
	/**
	 * Returns the current token sequence.
	 * 
	 * @return The current token sequence.
	 */
	public List<Terminal> getTokens() {
		return new ArrayList<Terminal>(tokens);
	}
	
	/**
	 * Returns all tokens that are allowed to follow the current token sequence according to the grammar.
	 * 
	 * @return The possible next tokens.
	 */
	public List<Terminal> nextTokens() {
		ArrayList<Terminal> terminals = new ArrayList<Terminal>();
		if (debug) System.err.print("LOOKING FORWARD:");
		for (Edge e : chart.getEdgesByEndPos(tokens.size(), true)) {
			if (!e.isActive()) continue;
			if (!(e.getNextActive() instanceof Terminal)) continue;
			
			Terminal t = (Terminal) e.getNextActive();
			if (!terminals.contains(t)) {
				if (debug) System.err.print(" " + t);
				terminals.add(t);
			}
		}
		if (debug) System.err.println();
		
		return terminals;
	}
	
	/**
	 * Returns a boolean array that describes which of the tokens of the current token sequence are
	 * accessible (true) and which are not (false) for the given next token.
	 * 
	 * @param nextToken The next token for which the tokens should be checked for accessibility.
	 * @return A boolean array where each element stands for one token of the token sequence.
	 */
	public boolean[] getAccessiblePositions(String nextToken) {
		boolean[] pos = new boolean[tokens.size()];
		ArrayList<ArrayList<Edge>> paths = new ArrayList<ArrayList<Edge>>();
		
		for (Edge e : chart.getEdgesByEndPosAndActCat(tokens.size(), nextToken, false)) {
			if (!e.isActive()) continue;
			
			ArrayList<Edge> partialPath = new ArrayList<Edge>();
			partialPath.add(e.deepCopy());
			collectActivePaths(0, partialPath, paths);
		}
		
		for (ArrayList<Edge> path : paths) {
			for (Edge edge : path) {
				scanForAccessiblePositions(edge, pos, new ArrayList<Edge>());
			}
		}
		
		return pos;
	}
	
	private void scanForAccessiblePositions(Edge edge, boolean[] pos, ArrayList<Edge> visitedEdges) {
		if (edge.getStartPos() == edge.getEndPos()) return;
		if (edge.getBody().length == 0 && edge.getHead() instanceof Terminal) {
			pos[edge.getStartPos()] = true;
			return;
		}
		
		for (Edge e : edge.getLinks()) {
			if (!e.isAccessible()) continue;
			if (visitedEdges.contains(e)) continue;
			visitedEdges.add(e);
			scanForAccessiblePositions(e, pos, visitedEdges);
		}
	}
	
	private void collectActivePaths(int startPos, ArrayList<Edge> partialPath, ArrayList<ArrayList<Edge>> paths) {
		Edge edge = partialPath.get(partialPath.size()-1);
		if (edge.getStartPos() == startPos) {
			paths.add(partialPath);
			return;
		}
		if (edge.getStartPos() < startPos) {
			return;
		}
		ArrayList<Edge> edgesToCheck = new ArrayList<Edge>();
		for (Edge e : chart.getEdgesByEndPosAndActCat(edge.getStartPos(), edge.getHead().getName(), false)) {
			Category[] newBody = new Category[e.getProgress()+1];
			System.arraycopy(e.getBody(), 0, newBody, 0, e.getProgress()+1);
			Edge ec = new Edge(e.getStartPos(), e.getEndPos(), e.getHead(), newBody, e.getProgress(), true);
			ec.addLinksFrom(e);
			boolean isNew = true;
			for (Edge ee : edgesToCheck) {
				if (ee.subsumes(ec)) {
					isNew = false;
					break;
				}
			}
			if (isNew) {
				for (Edge ee : edgesToCheck) {
					if (ec.subsumes(ee)) {
						edgesToCheck.remove(ee);
					}
				}
				edgesToCheck.add(ec);
			}
		}
		for (Edge e : edgesToCheck) {
			try {
				Edge eC = e.deepCopy();
				ArrayList<Edge> newPartialPath = copyEdgeList(partialPath);
				Edge newEdge = newPartialPath.get(partialPath.size()-1);
				newEdge.getHead().unify(eC.getNextActive());
				newPartialPath.add(eC);
				collectActivePaths(startPos, newPartialPath, paths);
			} catch (UnificationFailedException ex) {
				continue;
			}
		}
	}
	
	private ArrayList<Edge> copyEdgeList(ArrayList<Edge> edgeList) {
		ArrayList<Edge> edgeListCopy = new ArrayList<Edge>();
		HashMap<Integer, StringEntity> entities = new HashMap<Integer, StringEntity>();
		for (Edge edge : edgeList) {
			edgeListCopy.add(edge.deepCopy(entities));
		}
		return edgeListCopy;
	}
	
	private void init(Nonterminal category) {
		for (Rule rule : grammar.getRulesByHeadName(category.getName())) {
			try {
				Nonterminal categoryC = (Nonterminal) category.deepCopy();
				Rule ruleC = rule.deepCopy();
				categoryC.unify(ruleC.getHead());
				Edge edge = new Edge(0, 0, ruleC.getHead(), ruleC.getBody(), ruleC.isAccessible());
				chart.addEdge(edge);
				if (debug) System.err.println("INIT: " + rule + "  >>>  " + edge);
			} catch (UnificationFailedException ex) {
				continue;
			}
		}
	}
	
	private void completeAndPredict() {
		int c = 0;
		do {
			complete();
			c = predict();
		} while (c > 0);
	}
	
	private int predict() {
		int count = 0;
		for (Edge e : chart.getEdgesByEndPos(tokens.size(), true)) {
			Category cat = e.getNextActive();
			if (cat == null) continue;
			
			if (debug) System.err.println("PREDICT FOR EDGE: " + e);
			predict(cat);
		}
		for (Rule rule : grammar.getEpsilonRules()) {
			Edge edge = new Edge(tokens.size(), tokens.size(), rule.getHead(), rule.isAccessible());
			boolean isNewEdge = chart.addEdge(edge);
			if (isNewEdge) count++;
			if (debug) System.err.println("EDGE FOR EPSILON RULE: " + edge);
		}
		return count;
	}
	
	private void predict(Category category) {
		for (Rule rule : grammar.getRulesByHeadName(category.getName())) {
			if (rule.hasEmptyBody()) continue;
			
			try {
				Category categoryC = category.deepCopy();
				Rule ruleC = rule.deepCopy();
				categoryC.unify(ruleC.getHead());
				int p = tokens.size();
				Edge edge = new Edge(p, p, ruleC.getHead(), ruleC.getBody(), ruleC.isAccessible());
				if (debug) System.err.println("PREDICTOR: " + rule + "  >>>  " + edge);
				boolean isNewEdge = chart.addEdge(edge);
				if (isNewEdge) {
					if (debug) System.err.println("PREDICT FOR EDGE: " + edge);
					predict(edge.getNextActive());
				}
			} catch (UnificationFailedException ex) {
				continue;
			}
		}
	}
	
	private void complete() {
		for (Edge e : chart.getEdgesByEndPosAndActCat(tokens.size(), null, true)) {
			if (debug) System.err.println("COMPLETE FOR EDGE: " + e);
			complete(e.getStartPos(), e);
		}
	}
	
	private void complete(int pos, Edge passiveEdge) {
		Category category = passiveEdge.getHead();
		
		for (Edge edge : chart.getEdgesByEndPosAndActCat(pos, category.getName(), pos == tokens.size())) {
			if (!edge.isActive()) continue;
			
			try {
				Category categoryC = category.deepCopy();
				Edge edgeC = edge.deepCopy();
				categoryC.unify(edgeC.getNextActive());
				edgeC.step(tokens.size(), passiveEdge);
				if (debug) System.err.println("COMPLETOR: " + edge + "  >>>  " + edgeC);
				boolean isNewEdge = chart.addEdge(edgeC);
				if (isNewEdge && !edgeC.isActive()) {
					complete(edgeC.getStartPos(), edgeC);
				}
			} catch (UnificationFailedException ex) {
				continue;
			}
		}
	}

}
