// This file is part of the Attempto Java Packages.
// Copyright 2008, Attempto Group, University of Zurich (see http://attempto.ifi.uzh.ch).
//
// The Attempto Java Packages is free software: you can redistribute it and/or modify it under the
// terms of the GNU Lesser General Public License as published by the Free Software Foundation,
// either version 3 of the License, or (at your option) any later version.
//
// The Attempto Java Packages is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public License along with the Attempto
// Java Packages. If not, see http://www.gnu.org/licenses/.

package ch.uzh.ifi.attempto.chartparser;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Set;


class Edge {
	
	private int startPos;
	private int endPos;
	private Category head;
	private Category[] body;
	private int progress;
	private boolean accessible;
	private HashSet<Edge> links = new HashSet<Edge>();
	
	
	private Edge() {
	}
	
	public Edge(int startPos, int endPos, Category head, Category[] body, int progress, boolean accessible) {
		this.accessible = accessible;
		this.startPos = startPos;
		this.endPos = endPos;
		this.head = head;
		this.body = (Category[]) body.clone();
		this.progress = progress;
	}
	
	public Edge(int startPos, int endPos, Category head, Category[] body, boolean accessible) {
		this.accessible = accessible;
		this.startPos = startPos;
		this.endPos = endPos;
		this.head = head;
		this.body = (Category[]) body.clone();
		this.progress = 0;
	}
	
	public Edge(int startPos, int endPos, Category head, boolean accessible) {
		this.accessible = accessible;
		this.startPos = startPos;
		this.endPos = endPos;
		this.head = head;
		this.body = new Category[0];
		this.progress = 0;
	}
	
	public int getStartPos() {
		return startPos;
	}
	
	public int getEndPos() {
		return endPos;
	}
	
	public Category getHead() {
		return head;
	}
	
	public Category[] getBody() {
		return body;
	}
	
	public int getProgress() {
		return progress;
	}
	
	public Set<Edge> getLinks() {
		return links;
	}
	
	public boolean isActive() {
		return body.length > progress;
	}
	
	public boolean isAccessible() {
		return accessible;
	}
	
	public Category getNextActive() {
		if (!isActive()) return null;
		return body[progress];
	}
	
	public void step(int pos, Edge edge) {
		if (!isActive()) {
			throw new RuntimeException("Passive edge");
		}
		links.add(edge);
		progress++;
		endPos = pos;
	}
	
	public String toString() {
		String s = "<" + startPos + "," + endPos + "> " + head + " ";
		if (accessible) {
			s += "=>";
		} else {
			s += "~>";
		}
		for (int i=0 ; i < body.length ; i++) {
			if (i == progress) s += " .";
			s += " " + body[i];
		}
		if (!isActive()) s += " .";
		return s;
	}
	
	public Edge deepCopy() {
		return deepCopy(new HashMap<Integer, StringEntity>());
	}
	
	public Edge deepCopy(HashMap<Integer, StringEntity> entities) {
		Edge edgeC = new Edge();
		edgeC.startPos = startPos;
		edgeC.endPos = endPos;
		edgeC.progress = progress;
		edgeC.accessible = accessible;
		edgeC.head = head.deepCopy(entities);
		edgeC.body = new Category[body.length];
		for (int i=0 ; i < body.length ; i++) {
			edgeC.body[i] = body[i].deepCopy(entities);
		}
		edgeC.addLinksFrom(this);
		return edgeC;
	}
	
	public void addLinksFrom(Edge edge) {
		links = new HashSet<Edge>(edge.links);
	}
	
	public void skolemize() {
		head.skolemize();
		for (Category c : body) {
			c.skolemize();
		}
	}
	
	public void unify(Edge edge) throws UnificationFailedException {
		if (accessible != edge.accessible) {
			throw new UnificationFailedException();
		}
		if (body.length != edge.body.length) {
			throw new UnificationFailedException();
		}
		if (progress != edge.progress) {
			throw new UnificationFailedException();
		}
		head.unify(edge.head);
		for (int i = 0 ; i < body.length ; i++) {
			body[i].unify(edge.body[i]);
		}
	}
	
	public void tryToUnify(Edge edge) throws UnificationFailedException {
		if (accessible != edge.accessible) {
			throw new UnificationFailedException();
		}
		if (body.length != edge.body.length) {
			throw new UnificationFailedException();
		}
		if (progress != edge.progress) {
			throw new UnificationFailedException();
		}
		head.tryToUnify(edge.head);
		for (int i = 0 ; i < body.length ; i++) {
			body[i].tryToUnify(edge.body[i]);
		}
	}
	
	public boolean isSimilar(Edge edge) {
		if (accessible != edge.accessible) return false;
		if (startPos != edge.startPos) return false;
		if (endPos != edge.endPos) return false;
		if (!head.getName().equals(edge.head.getName())) return false;
		if (body.length != edge.body.length || progress != edge.progress) return false;
		for (int i = 0 ; i < body.length ; i++) {
			if (!body[i].getName().equals(edge.body[i].getName())) return false;
		}
		return true;
	}
	
	public boolean subsumes(Edge edge) {
		if (!isSimilar(edge)) return false;
		
		Edge edge1C = deepCopy();
		Edge edge2C = edge.deepCopy();
		edge2C.skolemize();
		try {
			edge1C.tryToUnify(edge2C);
			return true;
		} catch (UnificationFailedException ex) {
			return false;
		}
	}

}
