// This file is part of the Attempto Java Packages.
// Copyright 2008, Attempto Group, University of Zurich (see http://attempto.ifi.uzh.ch).
//
// The Attempto Java Packages is free software: you can redistribute it and/or modify it under the
// terms of the GNU Lesser General Public License as published by the Free Software Foundation,
// either version 3 of the License, or (at your option) any later version.
//
// The Attempto Java Packages is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public License along with the Attempto
// Java Packages. If not, see http://www.gnu.org/licenses/.

package ch.uzh.ifi.attempto.echocomp;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.UnsupportedEncodingException;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletInputStream;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletRequestWrapper;

/**
 * This filter is a workaround for the problem that Firefox 3.0.0 to 3.0.4 somehow sends wrong
 * XML headers to the server.
 * See the discussion in <a href="http://echo.nextapp.com/site/node/5230" target="_blank">this thread</a>
 * of the Echo forum.
 * This code is inspired by the workaround proposed by the user "Nadir".
 *<p>
 * In order to apply this filter, the following lines have to be added to web.xml:
 * 
 * <pre>
 * &lt;filter&gt;
 *   &lt;filter-name&gt;headercontrol&lt;/filter-name&gt;
 *   &lt;filter-class&gt;ch.uzh.ifi.attempto.echocomp.HeaderControlFilter&lt;/filter-class&gt;
 *   &lt;init-param&gt;
 *     &lt;param-name&gt;request.reencoding&lt;/param-name&gt;
 *     &lt;param-value&gt;UTF-8&lt;/param-value&gt;
 *   &lt;/init-param&gt;
 * &lt;/filter&gt;
 * 
 * &lt;filter-mapping&gt;
 *   &lt;filter-name&gt;headercontrol&lt;/filter-name&gt;
 *   &lt;url-pattern&gt;*&lt;/url-pattern&gt;
 * &lt;/filter-mapping&gt;
 * </pre>
 */
public class HeaderControlFilter implements Filter {
	
	protected String reencParam = null;
	
	public void init(FilterConfig filterConfig) throws ServletException {
		this.reencParam = filterConfig.getInitParameter("request.reencoding");
	}

	public void destroy() {
	}
	
	public void doFilter(ServletRequest req, ServletResponse resp, FilterChain chain) throws IOException, ServletException {
		HttpServletRequest httpRequest = (HttpServletRequest) req;
		// Firefox 3.0.4 requires a less strict check. Before it was:
		// if (reencParam != null && req.getCharacterEncoding() != null && !reencParam.equals(req.getCharacterEncoding())) {
		if (reencParam != null && req.getCharacterEncoding() != null) {
			req = new HeaderControlRequest(httpRequest, reencParam);
		}
		chain.doFilter(req, resp);
	}
	
}


class HeaderControlRequest extends HttpServletRequestWrapper {
	
	String encoding;
	ServletInputStream inputStream;
	
	public HeaderControlRequest(HttpServletRequest req, String encoding) {
		super(req);
		this.encoding = encoding;
	}
	
	public ServletInputStream getInputStream() throws IOException {
		if (inputStream == null) {
			inputStream = new InputStreamReencoder(super.getInputStream(), super.getCharacterEncoding(), encoding);
		}
		return inputStream;
	}
	
}


class InputStreamReencoder extends ServletInputStream {
	
	ByteArrayInputStream byteInputStream;
	
	public InputStreamReencoder(ServletInputStream inputStream, String encIn, String encOut) {
		ByteArrayOutputStream byteOutputStream = new ByteArrayOutputStream();
		byte[] buffer = new byte[4096];
		
		int bytesRead = 0;
		try {
			do {
				bytesRead = inputStream.read(buffer);
				if (bytesRead > 0) {
					byteOutputStream.write(buffer, 0, bytesRead);
				}
			} while (bytesRead > 0);
		} catch (IOException e) {
			e.printStackTrace();
		} finally {
			if (inputStream != null) {
				try {
					inputStream.close();
				} catch (IOException ex) {}
			}
		}
		
		byte[] data = byteOutputStream.toByteArray();
		try {
			data = new String(data, encIn).trim().getBytes(encOut);
			byteInputStream = new ByteArrayInputStream(data);
		} catch (UnsupportedEncodingException ex) {}
	}
	
	public int read() throws IOException {
		return byteInputStream.read();
	}

}
