// This file is part of the Attempto Java Packages.
// Copyright 2008, Attempto Group, University of Zurich (see http://attempto.ifi.uzh.ch).
//
// The Attempto Java Packages is free software: you can redistribute it and/or modify it under the
// terms of the GNU Lesser General Public License as published by the Free Software Foundation,
// either version 3 of the License, or (at your option) any later version.
//
// The Attempto Java Packages is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public License along with the Attempto
// Java Packages. If not, see http://www.gnu.org/licenses/.

package ch.uzh.ifi.attempto.preditor.text;

import java.util.ArrayList;
import java.util.List;

/**
 * This class represents a text container that stores a sequence of text elements that represent a
 * (partial) text or sentence.
 * 
 * @author Tobias Kuhn
 */
public class TextContainer {
	
	private ArrayList<TextElement> elements = new ArrayList<TextElement>();
	
	/**
	 * Creates a new text container.
	 */
	public TextContainer() {
	}
	
	/**
	 * Returns the number of text elements of this text container.
	 * 
	 * @return The number of text elements.
	 */
	public int getTextElementsCount() {
		return elements.size();
	}
	
	/**
	 * Returns the text element with the given index.
	 * @param index The index of the text element to be returned.
	 * @return The text element.
	 */
	public TextElement getTextElement(int index) {
		return elements.get(index);
	}
	
	/**
	 * Returns the sequence of text elements.
	 * 
	 * @return A list containing the text elements.
	 */
	public ArrayList<TextElement> getTextElements() {
		return new ArrayList<TextElement>(elements);
	}
	
	/**
	 * Sets the text elements.
	 * @param elements A list of text elements.
	 */
	public void setTextElements(List<TextElement> elements) {
		this.elements = new ArrayList<TextElement>(elements);
		updateConnections();
	}
	
	/**
	 * Adds the text element to the end of the sequence.
	 * 
	 * @param el The text element to be added.
	 */
	public void addElement(TextElement el) {
		elements.add(el);
		updateConnections();
	}
	
	/**
	 * Removes all text elements.
	 */
	public void removeAllElements() {
		elements.clear();
	}
	
	/**
	 * Removes the last text element of the sequence if it is not empty.
	 */
	public void removeLastElement() {
		if (elements.size() > 0) {
			elements.remove(elements.size() - 1);
			updateConnections();
		}
	}
	
	/**
	 * Returns the text that is represented by the sequence of text element as a string.
	 * 
	 * @return The text.
	 */
	public String getText() {
		String text = "";
		for (TextElement e : elements) {
			if (e.getText().matches("[.?!]")) {
				text += e.getText();
			} else {
				text += " " + e.getText();
			}
		}
		if (text.startsWith(" ")) {
			text = text.substring(1);
		}
		return text;
	}
	
	/**
	 * This method updates the connections between the text elements by calling their
	 * {@code checkNeighborTextElements}-method.
	 * 
	 * @see TextElement#checkNeighborTextElements
	 */
	public void updateConnections() {
		for (int i = 1; i < elements.size()-1; i++) {
			elements.get(i).checkNeighborTextElements(elements.get(i-1), elements.get(i+1));
		}
		if (elements.size() > 1) {
			elements.get(0).checkNeighborTextElements(null, elements.get(1));
			elements.get(elements.size()-1).checkNeighborTextElements(elements.get(elements.size()-2), null);
		}
		if (elements.size() == 1) {
			elements.get(0).checkNeighborTextElements(null, null);
		}
	}

}
